/* eslint-env node */

var inherit = require('inherit');
var assert = require('assert');
var config = require('../../configs/current');
var _ = require('lodash');
var LangSwitcherView = require('./LangSwitcherView');
var CompositeView = require('pview/CompositeView');
var HelpLinkView = require('../footer/HelpLinkView');
var NavigationView = require('../navigation/NavigationView');
var PView = require('pview');
var PController = require('pcontroller');
var PLog = require('plog');
var LANG_JS_RE = /\...\.js$/;
var dheaderSetup = require('../../routes/common/dheaderSetup');
var yaphoneLite = require('../../routes/common/yaphoneLite');
var getYaExperimentsFlags = require('../../routes/common/getYaExperimentsFlags');
const authCustomsVersion = require('../../routes/common/getCustomConfig')('customs.version.json').version;

const RetpathAndHostView = inherit(PView, {
    name: 'RetpathAndHostView',
    __constructor: function(controller) {
        this.__base.apply(this, arguments);
        this._controller = controller;
    },
    _compile: function() {
        const controller = this._controller;
        const retpathParam = this._controller.getRequestParam('retpath');
        const passportHost = controller.getHost();

        if (retpathParam) {
            return controller
                .validateRetpath(retpathParam)
                .then((retpath) => {
                    if (retpath) {
                        return {passportHost, retpath};
                    }

                    return {passportHost};
                })
                .catch((err) => {
                    new PLog(controller.getLogId(), 'passport', 'renderinglayout', 'RetpathAndHostView')
                        .warn()
                        .write('validateRetpath rejected', err);

                    return {passportHost};
                });
        }

        return {passportHost};
    }
});

const BackpathView = inherit(PView, {
    name: 'BackpathView',
    __constructor: function(controller) {
        this.__base.apply(this, arguments);
        this._controller = controller;
    },
    _compile: function() {
        const controller = this._controller;
        const backpathParam = this._controller.getRequestParam('backpath');

        if (backpathParam) {
            return controller
                .validateRetpath(backpathParam)
                .then((retpath) => {
                    if (retpath) {
                        return {retpath};
                    }

                    return {};
                })
                .catch((err) => {
                    new PLog(controller.getLogId(), 'passport', 'renderinglayout', 'BackpathView')
                        .warn()
                        .write('validateBackpath rejected', err);

                    return {};
                });
        }

        return {};
    }
});

var UatraitsView = inherit(PView, {
    name: 'UatraitsView',
    __constructor: function(controller) {
        this.__base.apply(this, arguments);
        this._controller = controller;
    },
    _compile: function() {
        var controller = this._controller;

        return controller
            .getUatraits()
            .then(function(uatraits) {
                return {
                    uatraits: uatraits
                };
            })
            .catch(function(err) {
                if (err && err.code !== 'need_resign') {
                    new PLog(controller.getLogId(), 'passport', 'renderinglayout', 'uatraitsview')
                        .warn()
                        .write('getUatraits rejected', err);
                }
                return {};
            });
    }
});

const DHeaderView = inherit(PView, {
    name: 'DHeaderView',
    __constructor(controller) {
        this.__base.apply(this, arguments);
        this._controller = controller;
    },
    _compile() {
        const controller = this._controller;
        const req = controller.getRequest();

        if (req.path.indexOf('/profile/access') !== 0) {
            return {};
        }

        // Простите меня
        return getYaExperimentsFlags(req, controller.getResponse(), () =>
            dheaderSetup(req, controller.getResponse(), () => ({
                experiments: controller.getLocalsField('experiments'),
                dheader: controller.getLocalsField('dheader')
            }))
        );
    }
});

const IsLiteView = inherit(PView, {
    name: 'IsLiteView',
    __constructor(controller) {
        this.__base.apply(this, arguments);
        this._controller = controller;
    },
    _compile() {
        const controller = this._controller;

        return yaphoneLite(controller.getRequest(), controller.getResponse(), () => ({
            isLite: controller.getLocalsField('isLite')
        }));
    }
});

var CsrfView = inherit(PView, {
    name: 'CsrfView',
    __constructor: function(controller) {
        this.__base.apply(this, arguments);
        this._controller = controller;
    },
    _compile: function() {
        var controller = this._controller;

        return controller
            .getCsrfToken()
            .then(function(csrf) {
                return {
                    csrf: csrf
                };
            })
            .catch(function(err) {
                if (err && err.code !== 'need_resign') {
                    new PLog(controller.getLogId(), 'passport', 'renderinglayout', 'csrfview')
                        .warn()
                        .write('getCsrfToken rejected', err);
                }
                return {};
            });
    }
});

var UserType = inherit(PView, {
    name: 'UserType',
    __constructor: function(controller) {
        this.__base.apply(this, arguments);
        this._controller = controller;
    },
    _compile: function() {
        var controller = this._controller;

        return controller
            .getAuth()
            .sessionID({
                multisession: 'yes',
                full_info: 'yes',
                get_family_info: 'yes',
                get_public_id: 'yes',
                allow_child: 'yes',
                emails: 'getdefault',
                // @see https://wiki.yandex-team.ru/passport/dbmoving/#tipyatributov
                attributes: '27,28,32,34,1003,1005,1011',
                aliases: '1,5,6,13,21'
            })
            .then(function(sessionInfo) {
                var defaultUser =
                    sessionInfo &&
                    _.filter(sessionInfo.users, function(user) {
                        return user.id === sessionInfo.default_uid;
                    })[0];
                var isWSUser;
                var isPDDUser;

                if (defaultUser) {
                    isWSUser = (defaultUser.attributes && Boolean(defaultUser.attributes[1011])) || false;
                    isPDDUser = (defaultUser.uid && defaultUser.uid.hosted) || false;
                }

                return {
                    isWSUser: isWSUser,
                    isPDDUser: isPDDUser
                };
            })
            .catch(function(err) {
                if (err && err.code !== 'need_resign') {
                    new PLog(controller.getLogId(), 'passport', 'renderinglayout', 'iswsuserview')
                        .warn()
                        .write('IsWSUser rejected', err);
                }
                return {};
            });
    }
});

var Metrics = require('../../routes/metrics').Metrics;
var metrics = new Metrics(config.metrics);
var MetricsIDView = inherit(PView, {
    name: 'MetricsIDView',
    __constructor: function(url) {
        assert(url && typeof url === 'string', 'Url should be a string');
        this._url = url;

        this.__base.apply(this, arguments);
    },
    _compile: function() {
        var counterID = metrics.getCounterID(this._url);

        return {
            metrics_id: counterID
        };
    }
});

var DefaultUserView = inherit(PView, {
    name: 'DefaultUserView',
    __constructor: function(controller) {
        this.__base.apply(this, arguments);
        this._controller = controller;
    },
    _compile: function() {
        var displayName;
        var avatar;
        var controller = this._controller;
        var auth = controller.getAuth();
        var authRequest = auth.getLastRequestPromise() || auth.loggedIn();

        return authRequest
            .then(function(sessionInfo) {
                var defaultUser;

                if (!sessionInfo) {
                    return {};
                }

                if (sessionInfo.users) {
                    defaultUser = _.filter(sessionInfo.users, function(user) {
                        return user.id === sessionInfo.default_uid;
                    })[0];
                } else {
                    defaultUser = sessionInfo;
                }

                displayName = defaultUser.display_name;

                if (!displayName) {
                    return {};
                }

                avatar = displayName.avatar || {};

                return {
                    defaultUser: {
                        displayName: displayName.name,
                        avatarUrl: avatar.empty ? '' : avatar.default
                    }
                };
            })
            .catch(function(err) {
                if (err && err.code !== 'need_resign') {
                    new PLog(controller.getLogId(), 'passport', 'renderinglayout', 'DefaultUserView')
                        .warn()
                        .write('blackbox data failed', err);
                }
                return {};
            });
    }
});

var UidView = inherit(PView, {
    name: 'UidView',
    __constructor: function(controller) {
        this.__base.apply(this, arguments);
        this._controller = controller;
    },
    _compile: function() {
        var auth = this._controller.getAuth();

        var authRequest = auth.getLastRequestPromise() || auth.loggedIn();

        var controller = this._controller;

        return authRequest
            .then(function() {
                var passportHost;

                switch (controller.getEnv()) {
                    case 'intranet':
                        passportHost = 'passport.yandex-team.';
                        break;

                    case 'testing':
                        passportHost = 'passport-test.yandex.';
                        break;

                    case 'rc':
                        passportHost = 'passport-rc.yandex.';
                        break;

                    case 'development':
                        var STRIP_TLD_RE = /(\.com\.tr|\.[^.]+)$/;

                        passportHost = controller.getUrl().hostname.replace(STRIP_TLD_RE, '.');
                        break;

                    default:
                        passportHost = 'passport.yandex.';
                }
                passportHost += controller.getTld();

                return {
                    ma: {
                        uid: config.multiauthPinning ? auth.getUid() : null,
                        login: config.multiauthPinning ? auth.getLogin() : null,
                        passportHost: passportHost
                    }
                };
            })
            .catch(function(err) {
                if (err && err.code !== 'need_resign') {
                    new PLog(controller.getLogId(), 'passport', 'renderinglayout', 'uidview')
                        .info()
                        .write('authRequest rejected', err);
                }
                return {};
            });
    }
});

/**
 * @class LayoutView
 * @typedef LayoutView
 * @extends CompositeView
 */
module.exports = inherit(CompositeView, {
    /**
     * A template file name
     * Should be defined by a successor view
     *
     * @example 'index.%lang%.js'
     * @type {string}
     */
    template: null,

    name: 'LayoutView',
    __constructor: function(controller) {
        assert(controller instanceof PController, 'Controller should be an instance of PController');

        this.__base.apply(this, arguments);
        this._controller = controller;
    },

    getTemplate: function(lang) {
        assert(this.template && typeof this.template === 'string', 'Template name should be defined');
        assert(typeof lang === 'string' && lang.length === 2, 'Lang should be a two-letter language code');
        return this.template.replace('%lang%', lang);
    },

    _compile: function(lang) {
        assert(typeof lang === 'string' && lang.length === 2, 'Lang should be a two-letter language code');

        if (!this._controller.getLocalsField('language')) {
            this._controller.setLocalsField('language', lang);
        }

        this.append(new MetricsIDView(this._controller.getUrl().pathname))
            .append(new DefaultUserView(this._controller))
            .append(new UatraitsView(this._controller))
            .append(new DHeaderView(this._controller))
            .append(new IsLiteView(this._controller))
            .append(new LangSwitcherView(this._controller))
            .append(new UserType(this._controller))
            .append(new NavigationView(this._controller))
            .append(new HelpLinkView(this._controller))
            .append(new CsrfView(this._controller))
            .append(new RetpathAndHostView(this._controller))
            .append(new UidView(this._controller))
            .append(new BackpathView(this._controller));

        var that = this;

        return this.__base.apply(this, arguments).then(function(compiled) {
            var controller = that._controller;
            var env = controller.getEnv();

            return _.assign(compiled, {
                logId: controller.getLogId(),
                env: env,
                domain: controller.getTld(),
                language: lang,
                origin: controller.getRequestParam('origin'),

                settings: {
                    env: env,
                    intranet: env === 'intranet',
                    page: that.getTemplate(lang).replace(LANG_JS_RE, ''),
                    authCustomsVersion: authCustomsVersion,
                    version: config.version,
                    paths: {
                        static: config.paths.static,
                        authCustomsStatic: config.paths.authCustomsStatic,
                        mda: config.paths.mda,
                        avatar: config.paths.avatar,
                        social_static: config.paths.social_static,
                        unread: config.paths.unread,
                        ysa: config.paths.ysa.static
                    }
                }
            });
        });
    },

    render: function() {
        var that = this;
        var render = this.__base;
        var controller = this._controller;

        return controller
            .getLanguage()
            .then(function(lang) {
                return render.call(that, lang);
            })
            .then(function(html) {
                if (controller._response && controller._response.headersSent) {
                    return;
                }

                return controller.sendPage(html);
            });
    }
});
