import api from '../react-blocks/api';

const MODIFICATORS = ['alt', 'ctrl', 'shift', 'meta'];
const KEYS_MAP = {
    '8': 'backspace',
    '9': 'tab',
    '13': 'enter',
    '32': 'space'
};

export default {
    init(opts) {
        const body = document.querySelector('body');
        const {extendedLogging} = body.dataset;
        const {trackId} = opts;

        this.trackId = trackId;
        this.onFocusInOut = this.onFocusInOut.bind(this);
        this.onKeyDownUp = this.onKeyDownUp.bind(this);

        if (extendedLogging) {
            this.trackUserActions();
        }
    },

    destroy() {
        const $document = $(document);

        if (document.removeEventListener) {
            document.removeEventListener('focus', this.onFocusInOut, true);
            document.removeEventListener('blur', this.onFocusInOut, true);
        } else {
            $document.off('focusin', this.onFocusInOut);
            $document.off('focusout', this.onFocusInOut);
        }

        $document.off('keyup', this.onKeyDownUp);
        $document.off('keydown', this.onKeyDownUp);
    },

    trackUserActions() {
        // focus/blur events don't bubble up, so we need to use focusin/focusout
        // but these events doesnt work in Firefox
        // @see https://bugzilla.mozilla.org/show_bug.cgi?id=687787
        const $document = $(document);

        if (document.addEventListener) {
            document.addEventListener('focus', this.onFocusInOut, true);
            document.addEventListener('blur', this.onFocusInOut, true);
        } else {
            $document.on('focusin', this.onFocusInOut);
            $document.on('focusout', this.onFocusInOut);
        }
        $document.on('keyup', this.onKeyDownUp);
        $document.on('keydown', this.onKeyDownUp);
    },
    /**
     * Listen to focus/blur events and log field name and its value length
     */
    onFocusInOut(event = {}) {
        const {target = {}, type} = event;
        const {name, value = '', tagName} = target;
        const eventType = type.match(/^focus$|^focusin$/) ? 'focus' : 'blur';

        if (tagName === 'INPUT') {
            api.log(
                {
                    action: 'field:' + eventType,
                    track_id: this.trackId,
                    withTrackId: true,
                    field: {
                        name,
                        length: value.length
                    }
                },
                {
                    encrypt: true
                }
            );
        }
    },

    /**
     * Listen to keydown/up event and log pressed keys with modificators
     * format of the action is "key[down|up]:[symbol|tab|space|backspace|enter]"
     * logging is disabled for password fields
     */
    onKeyDownUp(event = {}) {
        const {target, type, keyCode} = event;

        if (target.type === 'password' || (type === 'keydown' && this.isKeyDown)) {
            return;
        }

        const pressedModificators = [];

        let action = type + ':symbol';

        this.isKeyDown = type === 'keydown';

        if (keyCode in KEYS_MAP) {
            action = type + ':' + KEYS_MAP[keyCode];
        }

        MODIFICATORS.forEach((mod) => {
            if (event[`${mod}Key`]) {
                pressedModificators.push(mod);
            }
        });

        const logObj = {
            action,
            track_id: this.trackId,
            withTrackId: true
        };

        if (pressedModificators.length !== 0) {
            logObj.modificators = pressedModificators;
        }

        api.log(logObj, {
            encrypt: true
        });
    }
};
