import React from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import Header from '@components/Header/Header.jsx';
import Footer from '@blocks/authv2/components/Footer/Footer.jsx';
import {Button} from '@components/Button';
import {Field} from '@components/Field';
import Form from '@blocks/authv2/components/Form/Form.jsx';
import {errors} from './errors';

const b = cn('auth-device');

class AuthDevice extends React.PureComponent {
    constructor(props) {
        super(props);

        this.state = {
            code: ''
        };

        this.fieldOptions = {
            autoCorrect: 'off',
            autoCapitalize: 'off',
            autoComplete: 'off'
        };
    }

    componentDidMount() {
        if (this.codeField && this.codeField.focus) {
            this.codeField.focus();
        }
    }

    onFieldChange = (code) => {
        this.setState({code});
    };

    onSubmit = (event) => {
        if (event) {
            event.preventDefault();
        }

        const {submitDevice} = this.props;
        const {code} = this.state;

        if (submitDevice) {
            submitDevice(code);
        }
    };

    onCommit = (event) => {
        if (event) {
            event.preventDefault();
        }

        const {commitDevice} = this.props;
        const {code} = this.state;

        if (commitDevice) {
            commitDevice(code);
        }
    };

    onCancel = () => {
        const {clearDeviceState} = this.props;

        if (clearDeviceState) {
            clearDeviceState();
        }

        this.setState({code: ''});
    };

    getError = () => {
        const {error} = this.props;

        if (!error) {
            return '';
        }

        return errors[error] || errors.internal;
    };

    goToYandex = () => {
        const {tld = 'ru'} = this.props;

        location.href = `//yandex.${tld}`;
    };

    maybeRenderSuccess = () => {
        const {successAuth} = this.props;

        if (!successAuth) {
            return null;
        }

        return (
            <Form>
                <div className={b('access')}>
                    <Button
                        type='submit'
                        text={i18n('OneDomik.device.success')}
                        view='action'
                        width='max'
                        size='l'
                        onClick={this.goToYandex}
                    />
                </div>
            </Form>
        );
    };

    maybeRenderConfirmation = () => {
        const {shouldShowConfirmation} = this.props;

        if (!shouldShowConfirmation) {
            return null;
        }

        return (
            <Form>
                <div className={b('access')}>
                    <Button
                        type='submit'
                        text={i18n('OneDomik.device.confirm')}
                        view='action'
                        width='max'
                        size='l'
                        onClick={this.onCommit}
                    />

                    <Button
                        type='submit'
                        text={i18n('OneDomik.device.cancel')}
                        view='pseudo'
                        width='max'
                        size='l'
                        onClick={this.onCancel}
                    />
                </div>
            </Form>
        );
    };

    maybeRenderForm = () => {
        const {successAuth, shouldShowConfirmation, shouldShowRestart} = this.props;

        if (!successAuth && !shouldShowConfirmation && !shouldShowRestart) {
            return (
                <Form>
                    <div className={b('access')}>
                        <Field
                            size='l'
                            ref={(field) => {
                                this.codeField = field;
                            }}
                            label={i18n('_AUTH_.device.field.label')}
                            value={this.state.code}
                            name='code'
                            type='text'
                            error={this.getError()}
                            options={this.fieldOptions}
                            onChange={this.onFieldChange}
                            placeholder='XXXX XXX XXX'
                            view='floating-label'
                        />

                        <Button
                            type='submit'
                            text={i18n('OneDomik.device.submit')}
                            view='action'
                            width='max'
                            size='l'
                            onClick={this.onSubmit}
                        />
                    </div>
                </Form>
            );
        }

        return null;
    };

    maybeRenderRestart = () => {
        const {shouldShowRestart} = this.props;

        if (!shouldShowRestart) {
            return null;
        }

        return (
            <Form cls={b()}>
                <div className={b('access')}>
                    <Button
                        type='submit'
                        text={i18n('OneDomik.device.retry')}
                        view='action'
                        width='max'
                        size='l'
                        onClick={this.onCancel}
                    />
                </div>
            </Form>
        );
    };

    getTitle = () => {
        const {successAuth, shouldShowConfirmation, title, shouldShowRestart} = this.props;

        if (successAuth) {
            return i18n('OneDomik.device.enter.success');
        }

        if (shouldShowConfirmation) {
            return title;
        }

        if (shouldShowRestart) {
            return i18n('OneDomik.device.enter.retry');
        }

        return i18n('OneDomik.device.enter.code');
    };

    maybeRenderTitle = () => <h1 className={b('title')}>{this.getTitle()}</h1>;

    isAvailable = () => {
        const {access = {}} = this.props;
        const {is2faEnabled, isPasswordStrongPolicy} = access;

        return !is2faEnabled && !isPasswordStrongPolicy;
    };

    maybeRenderFormContent = () => {
        if (!this.isAvailable()) {
            return null;
        }

        return (
            <React.Fragment>
                <div className={b('form-header')}>{this.maybeRenderTitle()}</div>

                {this.maybeRenderForm()}
                {this.maybeRenderRestart()}
                {this.maybeRenderConfirmation()}
                {this.maybeRenderSuccess()}
            </React.Fragment>
        );
    };

    maybeRenderAccessInfo = () => {
        const {access = {}} = this.props;
        const {is2faEnabled, isPasswordStrongPolicy} = access;

        if (this.isAvailable()) {
            return null;
        }

        return (
            <Form>
                <div className={b('info')}>
                    {is2faEnabled && (
                        <React.Fragment>
                            <div>
                                <span className={b('qr-icon')} />
                            </div>
                            <p>{i18n('OneDomik.device.access.2fa.title')}</p>
                            <p>{i18n('OneDomik.device.access.2fa.description')}</p>
                        </React.Fragment>
                    )}

                    {isPasswordStrongPolicy && (
                        <React.Fragment>
                            <p>{i18n('OneDomik.device.access.password_strong_policy.title')}</p>
                            <p>{i18n('OneDomik.device.access.password_strong_policy.description')}</p>
                        </React.Fragment>
                    )}
                </div>
            </Form>
        );
    };

    render() {
        return (
            <div className={b()}>
                <Header hideNavigation={true} />

                <div className={b('form')}>
                    {this.maybeRenderAccessInfo()}
                    {this.maybeRenderFormContent()}
                </div>

                <Footer />
            </div>
        );
    }
}

AuthDevice.propTypes = {
    tld: PropTypes.string.isRequired,
    error: PropTypes.string,
    title: PropTypes.string,
    shouldShowConfirmation: PropTypes.bool,
    shouldShowRestart: PropTypes.bool,
    successAuth: PropTypes.bool,
    submitDevice: PropTypes.func.isRequired,
    commitDevice: PropTypes.func.isRequired,
    clearDeviceState: PropTypes.func.isRequired,
    access: PropTypes.shape({
        is2faEnabled: PropTypes.bool.isRequired,
        isPasswordStrongPolicy: PropTypes.bool.isRequired
    })
};

export {AuthDevice};
