import url from 'url';
import api from '@blocks/api';
import {updateTrack} from '@blocks/common/actions';
import {
    updateFormType,
    updateFormPrefix,
    updateValues,
    updateStates,
    updateValidationMethod,
    updateHumanConfirmationStatus
} from '@blocks/actions/form';
import {restoreStateAndRedirect} from '@blocks/auth/actions';
import {updatePerson} from '@blocks/authv2/actions';
import {FIELDS_NAMES} from '@components/Field/names';

const ERRORS_FOR_SET_RETPATH = [
    'action.not_required',
    'account.disabled',
    'account.global_logout',
    'sessionid.invalid',
    'sslsession.required'
];

export function prepareStore() {
    return async (dispatch, getState) => {
        const state = getState();
        const {person = {}, settings = {}, common = {}} = state;
        const {isSocialchik} = person;
        const {track_id, retpath} = common;
        const {host, isMobile} = settings;

        const data = {retpath};

        if (track_id) {
            data.track_id = track_id;
        }

        try {
            const result = await api.doCompleteSubmit(data);

            if (result && result.status === 'ok') {
                const phone = result.phone_number || result.phone_number_confirmed;
                const validationMethod = result['human-confirmation'];
                const isLiteWithoutPassword = result.state === 'complete_lite' && isSocialchik;
                const formType = result.state;

                // легаси магия
                const hasRecoveryMethod =
                    (result.has_recovery_method === undefined && validationMethod === undefined) ||
                    result.has_recovery_method;

                dispatch(updatePerson({hasRecoveryMethod, isLiteWithoutPassword}));
                dispatch(updateTrack(result.track_id));
                dispatch(updateFormType(formType));

                if ((formType === 'complete_lite' && hasRecoveryMethod) || formType === 'complete_neophonish') {
                    dispatch(updateValidationMethod(undefined));
                }

                const formPrefix = `${formType}_v3${isMobile ? '_mobile' : ''}`;

                dispatch(updateFormPrefix(formPrefix));

                if (phone && validationMethod === 'phone') {
                    dispatch(
                        updateValues({
                            field: FIELDS_NAMES.PHONE,
                            value: phone
                        })
                    );
                    dispatch(
                        updateStates({
                            field: FIELDS_NAMES.PHONE,
                            status: 'valid'
                        })
                    );
                    dispatch(updateHumanConfirmationStatus(true));
                }

                return;
            }

            throw result;
        } catch (errResult) {
            const err = errResult.errors;
            const currentHost = url.parse(host, true);
            const currentUrl = retpath && url.parse(retpath, true);
            const defaultRetpathObj = {
                protocol: currentHost.protocol,
                hostname: currentHost.hostname,
                pathname: 'passport',
                query: {
                    mode: 'passport'
                }
            };
            const retpathToComplete = defaultRetpathObj;

            if (Array.isArray(err) && err.some((error) => ERRORS_FOR_SET_RETPATH.includes(error))) {
                if (err.some((error) => ['sessionid.invalid', 'sslsession.required'].includes(error))) {
                    retpathToComplete.pathname = 'auth';
                    retpathToComplete.query.retpath =
                        currentUrl &&
                        url.format(
                            Object.assign({}, defaultRetpathObj, {
                                pathname: currentUrl.pathname,
                                query: currentUrl.query,
                                search: null
                            })
                        );
                    delete retpathToComplete.query.mode;
                }

                if (err.includes('sslsession.required')) {
                    retpathToComplete.pathname = 'auth/secure';
                }
            }

            if (retpathToComplete.query.hasOwnProperty('retpath') && !retpathToComplete.query.retpath) {
                delete retpathToComplete.query.retpath;
            }

            await dispatch(restoreStateAndRedirect(url.format(retpathToComplete)));
        }
    };
}
