import React from 'react';
import PropTypes from 'prop-types';
import {Spin} from '@components/Spin';
import {Button} from '@components/Button';
import {cn} from '@bem-react/classname';
import errors from '@blocks/authv2/errors';
import {sendMessageUnsafe} from '@blocks/utils';
import {NoQrBgSvg} from './NoQrBgSvg';
import './AuthSmartTV.styl';

const SHOW_SKIP_BUTTON_TIMEOUT = 40 * 1000;
const SKIP_RETPATH = '/closewebview/cancel';

const formatUserCode = (userCode = '') => {
    if (!userCode) {
        return '';
    }

    return [userCode.slice(0, 4), userCode.slice(4, 7), userCode.slice(7, 10)].join(' ');
};

const b = cn('PassportSmarttv');

class AuthSmartTV extends React.PureComponent {
    constructor(props) {
        super(props);

        this.state = {
            trackId: props.trackId || '',
            csrfToken: props.csrfToken || '',
            error: props.error || '',
            retpath: props.retpath,
            is403ErrorScreen: false,
            userCode: formatUserCode(props.userCode),
            isNoQrScreen: props.isNoQrScreen,
            showSkipButton: false,
            isSecondDesignVersion: props.isSecondDesignVersion
        };
    }

    componentDidMount() {
        this.restart();

        if (this.props.willShowSkipButton) {
            setTimeout(() => {
                this.setState({showSkipButton: true});
            }, SHOW_SKIP_BUTTON_TIMEOUT);
        }

        $(document).on('ajaxError', (event, xhr) => {
            if (xhr.status === 403) {
                this.setState({is403ErrorScreen: true});
            }
        });
    }

    componentDidUpdate(prevProps) {
        if (prevProps.csrfToken !== this.props.csrfToken) {
            this.restart();
            return;
        }

        if (prevProps.trackId !== this.props.trackId) {
            this.restart();
        }
    }

    restart() {
        this.trackNotFoundErrorCount = 0;
        this.timeoutErrorCount = 0;

        this.stop();

        this._interval = 300;

        window.intervalTimeout = setTimeout(() => {
            this._interval = 1000;
        }, 60 * 1000);

        window.redirectTimeout = setTimeout(() => {
            window.location.reload();
        }, 60 * 1000 * 10);

        this._stopped = false;
        this.restartPolling();
    }

    restartPolling() {
        if (!this._stopped) {
            setTimeout(() => {
                this.poll();
            }, this._interval);
        }
    }

    sendMessage = (msg) => {
        sendMessageUnsafe({smartTvAuth: msg});
    };

    poll() {
        const {trackId, csrfToken, isNoQrScreen} = this.state;

        const req = $.ajax('/auth/new/magic/status/', {
            data: {
                track_id: trackId,
                csrf_token: csrfToken
            },
            dataType: 'json',
            timeout: 10000,
            type: 'POST'
        });

        req.fail(() => {
            if (this.timeoutErrorCount > 1) {
                this.stop();
                this.setState({error: 'global'});
                this.sendMessage('error');
            }

            this.timeoutErrorCount++;
            this.restartPolling();
        });

        req.done((results) => {
            let error;

            if (results.status === 'ok' && results.state === 'otp_auth_finished') {
                this.stop();
                this.sendMessage('success');
                window.location.href = this.state.retpath;
                return;
            }

            if (results.state === 'auth_challenge') {
                this.stop();
                this.sendMessage('challenge');
                if (isNoQrScreen) {
                    window.location.reload();
                } else if (window.location.search) {
                    window.location.search += '&noQr=1';
                } else {
                    window.location.search = '?noQr=1';
                }
                return;
            }

            if (results.errors) {
                error = results.errors[0];

                if (error === 'password.not_matched') {
                    error = 'password.not_matched_2fa';
                }

                if (error === 'captcha.required') {
                    error = 'internal';
                }

                if (error === 'track.not_found') {
                    // TODO: убрать костыль для PASSP-21000
                    if (this.trackNotFoundErrorCount > 5) {
                        this.reAuthPasswordSubmit();
                    }

                    this.trackNotFoundErrorCount++;
                }

                if (
                    [
                        'internal.temporary',
                        'backend.redis_failed',
                        'oauth_token.invalid',
                        'oauth_code.not_found'
                    ].includes(error)
                ) {
                    this.reAuthPasswordSubmit();
                }

                this.stop();
                this.setState({error});
                this.sendMessage('error');
            }

            this.restartPolling();
        });
    }

    stopPoll() {
        this._stopped = true;
    }

    stop() {
        this.stopPoll();

        clearTimeout(window.intervalTimeout);
        clearTimeout(window.redirectTimeout);
    }

    reAuthPasswordSubmit() {
        const {common = {}} = this.props;
        const {csrf} = common;

        const req = $.ajax('/registration-validations/reAuthPasswordSubmit', {
            data: {
                csrf_token: csrf
            },
            dataType: 'json',
            type: 'POST'
        });

        req.fail(() => {
            setTimeout(() => {
                this.reAuthPasswordSubmit();
            }, 3000);
        });

        req.done(({csrf_token: csrf, track_id: trackId, user_code: userCode}) => {
            this.setState({
                error: '',
                csrfToken: csrf,
                trackId,
                userCode: formatUserCode(userCode)
            });
            this.restart();
        });
    }

    maybeRenderError = () => {
        const {error, isSecondDesignVersion} = this.state;

        if (!error) {
            return null;
        }

        let message = errors[error] || errors.internal;

        let addMessage = '';

        if (isSecondDesignVersion && message === errors.internal) {
            const msgArr = message.split(':');

            message = msgArr[0];
            addMessage = msgArr[1];
        } else if (isSecondDesignVersion) {
            const errorTextMap = {
                'account.invalid_type': {
                    message: i18n('Frontend.smarttv.errors.account.invalid_type.message'),
                    addMessage: i18n('Frontend.smarttv.errors.account.invalid_type.addMessage')
                }
            };

            if (errorTextMap[error] && errorTextMap[error].message) {
                message = errorTextMap[error].message;
            }
            if (errorTextMap[error] && errorTextMap[error].addMessage) {
                addMessage = errorTextMap[error].addMessage;
            }
        }

        return isSecondDesignVersion ? (
            <div className={b('error', {v2: isSecondDesignVersion})}>
                <div className={b('errorMessage')}>{message}</div>
                {addMessage && <div className={b('errorAddMessage')}>{addMessage}</div>}
                {this.renderButtons()}
            </div>
        ) : (
            <div className={b('error')} dangerouslySetInnerHTML={{__html: message}} />
        );
    };

    render403ErrorScreen = () => {
        const {isSecondDesignVersion} = this.state;

        return (
            <>
                <h1 className={b('errorTitle', {v2: isSecondDesignVersion})}>{i18n('Frontend.smarttv.error-title')}</h1>
                <div className={b('errorButtonWrap', {v2: isSecondDesignVersion})}>
                    <Button
                        className={b('button', {v2: isSecondDesignVersion})}
                        onClick={this.onClickReloadButton}
                        innerRef={this.setButtonFocused}
                        view='auth-smart-tv'
                        type='submit'
                        width='max'
                        size='smarttv-xl'
                    >
                        {i18n('Frontend.reload')}
                    </Button>
                </div>
            </>
        );
    };

    renderQr = () => {
        const {trackId, isSecondDesignVersion} = this.state;

        return (
            <div className={b('qrWrap', {v2: isSecondDesignVersion})}>
                {trackId ? (
                    <img
                        className={b('qrImg', {v2: isSecondDesignVersion})}
                        src={`/auth/magic/code/?track_id=${trackId}`}
                    />
                ) : (
                    <Spin progress={true} size='m' />
                )}
            </div>
        );
    };

    getIconHtml = (name) => {
        const {settings = {}} = this.props;
        const {staticPath} = settings;

        return `<img class="${b('qrIcon')}" src="${staticPath}/i/${name}.svg" />`;
    };

    renderButtons = () => {
        const {showSkipButton, error, isNoQrScreen, isSecondDesignVersion} = this.state;
        const showErrorButton = ['global', 'internal', 'auth.challenge_tv', 'account.invalid_type'].includes(error);
        const showButtons = showSkipButton || showErrorButton;
        const isAbsolute = isSecondDesignVersion && !error;

        if (!showButtons) {
            return null;
        }

        return (
            <div className={b('buttons', {noQr: isNoQrScreen, v2: isSecondDesignVersion, isAbsolute})}>
                {showErrorButton && (
                    <Button
                        className={b('button', {v2: isSecondDesignVersion})}
                        view='auth-smart-tv'
                        type='submit'
                        size='smarttv-l'
                        onClick={this.onClickReloadButton}
                        innerRef={this.setButtonFocused}
                    >
                        {i18n('Frontend.reload')}
                    </Button>
                )}
                {showSkipButton && (
                    <Button
                        className={b('button', {v2: isSecondDesignVersion})}
                        view='auth-smart-tv'
                        type='submit'
                        size='smarttv-l'
                        onClick={this.onClickSkipButton}
                        innerRef={this.setButtonFocused}
                    >
                        {i18n('Frontend.skip')}
                    </Button>
                )}
            </div>
        );
    };

    renderQrBlock = () => {
        const {isSecondDesignVersion, isNoQrScreen} = this.state;
        const {settings = {}} = this.props;
        const {lang} = settings;
        const YaIconPostfix = ['ru', 'by', 'uk'].includes(lang) ? 'ru' : 'en';

        return isSecondDesignVersion ? (
            <div className={b('qr', {noQr: isNoQrScreen, v2: true})}>
                {isNoQrScreen && (
                    <>
                        <div className={b('qrTitle', {noQr: isNoQrScreen, v2: true})}>
                            {i18n('Frontend.smarttv.title-no-qr.v2')}
                        </div>
                        <div className={b('qrLabel', {noQr: isNoQrScreen, v2: true})}>
                            {i18n('Frontend.smarttv.subtitle-no-qr.v2')}
                        </div>
                    </>
                )}

                {!isNoQrScreen && (
                    <>
                        <div className={b('qrTitle', {noQr: isNoQrScreen, v2: true})}>
                            {i18n('Frontend.smarttv.qr-title.v2')}
                        </div>
                        <div
                            className={b('qrLabel', {noQr: isNoQrScreen, v2: true})}
                            dangerouslySetInnerHTML={{
                                __html: i18n('Frontend.smarttv.qr.any_reader-title').replace(
                                    '%s',
                                    this.getIconHtml('smarttv/yandex.key')
                                )
                            }}
                        />

                        {this.renderQr()}
                    </>
                )}
            </div>
        ) : (
            <div className={b('qr')}>
                <div
                    className={b('label')}
                    dangerouslySetInnerHTML={{
                        __html: i18n('Frontend.smarttv.qr-title')
                            .replace('%s', this.getIconHtml(`smarttv/YaEllips_${YaIconPostfix}`))
                            .replace('%s', this.getIconHtml('camera-icon'))
                    }}
                />
                {this.renderQr()}
                <div className={b('qrLabel')}>{i18n('Frontend.smarttv.qr-2fa')}</div>
            </div>
        );
    };

    renderCodeBlock = () => {
        const {settings = {}} = this.props;
        const {tld} = settings;
        const {userCode, isNoQrScreen, isSecondDesignVersion} = this.state;

        return isSecondDesignVersion ? (
            <div className={b('code', {v2: true})}>
                <div className={b('codeMain', {v2: true})}>
                    <div className={b('codeTitle', {noQr: isNoQrScreen, v2: true})}>
                        {i18n('Frontend.smarttv.code-title.v2')}
                    </div>
                    <div className={b('codeLabel', {noQr: isNoQrScreen, v2: true})}>
                        {i18n('Frontend.smarttv.code-go-to-page.v2')}
                        &nbsp;
                        <b>{`yandex.${tld}/activate`}</b>
                    </div>
                </div>

                <div className={b('codeFooter', {noQr: isNoQrScreen, v2: true})}>
                    <div className={b('codeLink', {noQr: isNoQrScreen, v2: true})}>
                        {isNoQrScreen && <div className={b('codeArrow', {v2: true})} />}
                        {`yandex.${tld}/activate`}
                    </div>
                    <div className={b('codeLabel', {alignLeft: true, noQr: isNoQrScreen, v2: true})}>
                        {i18n('Frontend.smarttv.code-enter.v2')}
                    </div>
                    <div className={b('codeField', {noQr: isNoQrScreen, v2: true})}>{userCode}</div>
                </div>
            </div>
        ) : (
            <div className={b('code', {noQr: isNoQrScreen})}>
                {isNoQrScreen && (
                    <h1 className={b('mainTitle', {noQr: true})}>{i18n('Frontend.smarttv.main-title-no-qr.v2')}</h1>
                )}
                <div className={b('codeMain')}>
                    <div className={b('label', {noQr: isNoQrScreen})}>
                        {isNoQrScreen
                            ? i18n('Frontend.smarttv.code-go-to-page-no-qr.v2')
                            : i18n('Frontend.smarttv.code-go-to-page')}
                    </div>
                    <div className={b('codeItem', {noQr: isNoQrScreen})}>{`yandex.${tld}/activate`}</div>

                    <div className={b('label', {noQr: isNoQrScreen})}>{i18n('Frontend.smarttv.code-enter')}</div>
                    <div className={b('codeItem', {noQr: isNoQrScreen})}>{userCode}</div>
                </div>
                <div className={b('codeFooter', {noQr: isNoQrScreen})}>
                    {!isNoQrScreen && this.maybeRenderError()}

                    {this.renderButtons()}

                    {isNoQrScreen && this.maybeRenderError()}
                </div>
            </div>
        );
    };

    renderClassicDesign = () => {
        const {is403ErrorScreen, isNoQrScreen} = this.state;

        return (
            <div className={b({noQr: isNoQrScreen})}>
                {is403ErrorScreen ? (
                    this.render403ErrorScreen()
                ) : (
                    <>
                        {!isNoQrScreen && <h1 className={b('mainTitle')}>{i18n('Frontend.smarttv.main-title')}</h1>}
                        <div className={b('content', {noQr: isNoQrScreen})}>
                            {isNoQrScreen ? this.renderCodeBlock() : this.renderQrBlock()}
                            <div className={b('separator', {hidden: isNoQrScreen})} />
                            {isNoQrScreen ? (
                                <div className={b('bg')}>
                                    <NoQrBgSvg />
                                </div>
                            ) : (
                                this.renderCodeBlock()
                            )}
                        </div>
                    </>
                )}
            </div>
        );
    };

    renderSecondDesign = () => {
        const {is403ErrorScreen, isNoQrScreen, error} = this.state;

        return (
            <div className={b({noQr: isNoQrScreen, v2: true})}>
                {is403ErrorScreen ? (
                    this.render403ErrorScreen()
                ) : (
                    <>
                        <div className={b('content', {noQr: isNoQrScreen, v2: true})}>
                            {this.renderQrBlock()}

                            {this.renderCodeBlock()}

                            {this.maybeRenderError()}

                            {!error && this.renderButtons()}
                        </div>
                    </>
                )}
            </div>
        );
    };

    onClickReloadButton = () => {
        window.location.reload();
    };

    onClickSkipButton = () => {
        window.location.href = SKIP_RETPATH;
    };

    setButtonFocused = (button) => {
        button && button.focus();
    };

    render() {
        const {isSecondDesignVersion} = this.state;

        return isSecondDesignVersion ? this.renderSecondDesign() : this.renderClassicDesign();
    }
}

AuthSmartTV.propTypes = {
    settings: PropTypes.shape({
        lang: PropTypes.string.isRequired,
        tld: PropTypes.string.isRequired,
        staticPath: PropTypes.string.isRequired
    }),
    isNoQrScreen: PropTypes.bool,
    retpath: PropTypes.string.isRequired,
    error: PropTypes.string,
    trackId: PropTypes.string,
    csrfToken: PropTypes.string,
    userCode: PropTypes.string.isRequired,
    common: PropTypes.shape({
        csrf: PropTypes.string.isRequired
    }),
    willShowSkipButton: PropTypes.bool.isRequired,
    isSecondDesignVersion: PropTypes.bool.isRequired
};

export {AuthSmartTV};
