import './AccountsListItem.styl';
import React from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import metrika from '@blocks/metrics';
import socialAliases from '@blocks/authv2/social_aliases';

const b = cn('AccountsListItem');

class AccountsListItem extends React.PureComponent {
    onAccountClick = (params) => {
        const {onClick} = this.props;

        if (typeof onClick === 'function') {
            metrika.send('Выбор аккаунта для авторизации');
            onClick(params);
        }
    };

    renderButton = () => {
        const {onClick, text, testKey} = this.props;

        return (
            <button onClick={onClick} className={b('account')} type='button' data-t={testKey} id={testKey}>
                <span className={b('avatar', {add: true})} />
                <span className={b('info')}>
                    <span className={b('name')}>{text}</span>
                </span>
            </button>
        );
    };

    renderTags = (hasPlus, hasCard, hasFamily) => {
        if (hasFamily || hasCard || hasPlus) {
            return (
                <ul className={b('tags')}>
                    {hasPlus && <li className={b('tag', {icon: 'plus'})}>{i18n('_AUTH_.plus.tag')}</li>}
                    {hasCard && <li className={b('tag', {icon: 'cards'})}>{i18n('_AUTH_.cards.tag')}</li>}
                    {hasFamily && <li className={b('tag', {icon: 'family'})}>{i18n('_AUTH_.family.tag')}</li>}
                </ul>
            );
        }

        return null;
    };

    render() {
        const {account, isButton, isStatic, testKey, defaultAvatarUrl, useNewSuggestByPhone} = this.props;

        if (isButton) {
            return this.renderButton();
        }

        const Tag = isStatic ? 'span' : 'a';
        const {
            login,
            name: accountName,
            display_name: displayName = {},
            default_avatar: defaultAvatar,
            avatar_url: avatarUrl,
            has_plus: hasPlus = false,
            has_bank_card: hasCard,
            has_family: hasFamily,
            isSocial,
            phone_number: phoneNumber = {}
        } = account;
        const name = displayName.name || accountName;
        const {masked_international: phone} = phoneNumber;
        const {social: {provider} = {}} = displayName;

        const props = isStatic
            ? {}
            : {
                  href: `/auth?login=${login}`,
                  target: '_blank',
                  rel: 'noopener noreferrer',
                  onClick: (event) => this.onAccountClick({event, account})
              };

        const style =
            defaultAvatar === '0/0-0' || !avatarUrl
                ? {
                      backgroundImage: `url(${defaultAvatarUrl})`
                  }
                : {
                      backgroundImage: `url(${avatarUrl})`
                  };
        const hasTags = hasPlus || hasCard || hasFamily;

        return (
            <Tag
                {...props}
                className={b('account', {
                    isStatic,
                    isNewSuggest: useNewSuggestByPhone,
                    hasTags: useNewSuggestByPhone && hasTags
                })}
                data-t={testKey}
                id={testKey}
                aria-label={`${name ? `${i18n('_AUTH_.field_firstname')} ${name}` : ''}, ${
                    login || phone ? `${i18n('_AUTH_.ariaLabel.login')} ${login || phone}` : ''
                }`}
            >
                <span
                    className={b('avatar', {isSocial: isSocial || Boolean(provider), provider, hasPlus})}
                    style={style}
                    aria-hidden='true'
                />
                <div className={b('info')} aria-hidden='true'>
                    {name && <span className={b('name')}>{name}</span>}
                    {(provider || login || phone) && (
                        <span className={b('login')}>{socialAliases[provider] || login || phone}</span>
                    )}
                    {useNewSuggestByPhone && this.renderTags(hasPlus, hasCard, hasFamily)}
                </div>
            </Tag>
        );
    }
}

AccountsListItem.propTypes = {
    account: PropTypes.shape({
        login: PropTypes.string.isRequired,
        name: PropTypes.string,
        default_avatar: PropTypes.string,
        avatar_url: PropTypes.string,
        uid: PropTypes.number,
        isSocial: PropTypes.bool,
        provider: PropTypes.string,
        allowed_auth_flows: PropTypes.array,
        primary_alias_type: PropTypes.number,
        display_name: PropTypes.shape({
            name: PropTypes.string
        }),
        has_plus: PropTypes.bool,
        has_bank_card: PropTypes.bool,
        has_family: PropTypes.bool,
        phone_number: PropTypes.shape({
            masked_international: PropTypes.string
        })
    }),
    isStatic: PropTypes.bool,
    useNewSuggestByPhone: PropTypes.bool,
    onClick: PropTypes.func,
    isButton: PropTypes.bool,
    text: PropTypes.string,
    testKey: PropTypes.string,
    defaultAvatarUrl: PropTypes.string.isRequired
};

export {AccountsListItem};
