import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';

import './AuthLoginInputToggle.styl';
import {Button} from '@plibs/pcomponents/Button';
import LoginField from '@blocks/authv2/components/LoginField/LoginField.jsx';
import {Phone as PhoneField} from '@components/Phone';
import {PHONE_PLACEHOLDERS} from '@blocks/actions/phone';
import {FIELDS_NAMES} from '@components/Field/names';
import metrika from '@blocks/metrics';

const b = cn('AuthLoginInputToggle');

const METRIKA_PREFIX = 'Переключатель логин-телефон';
const BUTTONS_ORDER = {
    loginFirst: ['login', 'phone'],
    phoneFirst: ['phone', 'login']
};
const TEXTS = {
    login: {
        button: i18n('_AUTH_.social-accounts-register-lite.field.label.email')
    },
    phone: {
        button: i18n('_AUTH_.CheckData.phone')
    }
};
const METRIKA_TEXTS = {
    login: 'логин',
    phone: 'телефон'
};

export class AuthLoginInputToggle extends Component {
    componentDidMount() {
        const {fieldType} = this.props;

        metrika.send([`${METRIKA_PREFIX}: показ, включен ${METRIKA_TEXTS[fieldType]}`]);
    }

    toggleInput = (e) => {
        const {onChangeLogin, fieldType, toggleLoginFieldType, clearFieldError} = this.props;
        const {currentTarget: {dataset: {type} = {}} = {}} = e;

        if (fieldType !== type) {
            onChangeLogin('');
            clearFieldError();

            toggleLoginFieldType(type);
            metrika.send([`${METRIKA_PREFIX}: переключение, включили ${METRIKA_TEXTS[type]}`]);
        }
    };

    changePhoneField = (value) => {
        const {updateValue} = this.props;
        const PHONE_REGEXP = /^[-+()0-9\s]*$/;

        if (PHONE_REGEXP.test(value)) {
            updateValue(value);
        }
    };

    getButton = (type = 'login') => {
        const {fieldType} = this.props;

        return (
            <Button
                size='l'
                view={type === fieldType ? 'default' : 'clear'}
                type='button'
                text={TEXTS[type].button}
                onClick={this.toggleInput}
                data-type={type}
            />
        );
    };

    renderButtons = () => {
        const {isLoginMainExp} = this.props;

        const order = isLoginMainExp ? BUTTONS_ORDER.loginFirst : BUTTONS_ORDER.phoneFirst;

        return (
            <div className={b('wrapper')}>
                {order.map((type) => (
                    <div key={type} className={b('type')}>
                        {this.getButton(type)}
                    </div>
                ))}
            </div>
        );
    };

    renderInput = () => {
        const {login, phone, fieldType, onChangeLogin, tld, hasRoundViewExp, error} = this.props;

        switch (fieldType) {
            case 'phone':
                if (hasRoundViewExp) {
                    return (
                        <div className={b('input')}>
                            <PhoneField
                                hideLabel={true}
                                error={error}
                                name={FIELDS_NAMES.PHONE}
                                hasPhoneMask={true}
                                view='rounded-corner'
                                size='l'
                            />
                        </div>
                    );
                }

                return (
                    <div className={b('input')}>
                        <LoginField
                            customPlaceholder={tld === 'ru' ? '+7 123 456-78-90' : PHONE_PLACEHOLDERS[tld] || ''}
                            login={phone}
                            skipLink={true}
                            skipLabel={true}
                            onChangeLogin={this.changePhoneField}
                            fieldType='tel'
                        />
                    </div>
                );

            case 'login':
                return (
                    <div className={b('input')}>
                        <LoginField
                            customPlaceholder={i18n('_AUTH_.login_email.placeholder')}
                            login={login}
                            skipLabel={true}
                            onChangeLogin={onChangeLogin}
                        />
                    </div>
                );
        }
    };

    render() {
        return (
            <>
                {this.renderButtons()}
                {this.renderInput()}
            </>
        );
    }
}

AuthLoginInputToggle.propTypes = {
    login: PropTypes.string,
    phone: PropTypes.string,
    onChangeLogin: PropTypes.func.isRequired,
    clearFieldError: PropTypes.func.isRequired,
    updateValue: PropTypes.func.isRequired,
    tld: PropTypes.string,
    fieldType: PropTypes.string,
    isLoginMainExp: PropTypes.bool,
    hasRoundViewExp: PropTypes.bool,
    toggleLoginFieldType: PropTypes.func.isRequired,
    error: PropTypes.string
};
