import React from 'react';
import PropTypes from 'prop-types';
import {Select} from '@components/Select';
import {Input} from '@components/Input';
import dayjs from 'dayjs';
import {cn} from '@bem-react/classname';
import './BirthdayField.styl';

const b = cn('BirthdayField');

const createMonthOption = (content, value) => ({content, value});

const MONTHS_OPTIONS = [
    createMonthOption(i18n('Profile.personal.data.birthday.month'), ''),
    createMonthOption(i18n('Months.n1'), '01'),
    createMonthOption(i18n('Months.n2'), '02'),
    createMonthOption(i18n('Months.n3'), '03'),
    createMonthOption(i18n('Months.n4'), '04'),
    createMonthOption(i18n('Months.n5'), '05'),
    createMonthOption(i18n('Months.n6'), '06'),
    createMonthOption(i18n('Months.n7'), '07'),
    createMonthOption(i18n('Months.n8'), '08'),
    createMonthOption(i18n('Months.n9'), '09'),
    createMonthOption(i18n('Months.n10'), '10'),
    createMonthOption(i18n('Months.n11'), '11'),
    createMonthOption(i18n('Months.n12'), '12')
];

const TYPES = {
    default: 'default',
    kiddish: 'kiddish'
};

const getIsOldByType = (type, year) => {
    switch (type) {
        case TYPES.default:
            return year < 1900;
        case TYPES.kiddish:
            return year < 1970;
        default:
            return year < 1900;
    }
};

const getDayStr = (day) => `${day.toString().length < 2 ? '0' : ''}${day}`;

const dataValidation = ({day = '', month = '', year = '', type} = {}) => {
    let [isDayIncorrect, isDateInFuture, isOld, hasSymbols] = [false, false, false, false];

    const daysInMonth = dayjs()
        .year(year)
        .month(month - 1)
        .daysInMonth();
    const dayStr = getDayStr(day);

    isDayIncorrect = day < 1 || day > daysInMonth || dayStr.length > 2;
    isDateInFuture = dayjs().isBefore(
        dayjs()
            .year(year)
            .month(month - 1)
            .date(day)
    );
    isOld = getIsOldByType(type, year);
    hasSymbols = [day, month, year].some((str) => /\D/.test(str));

    return isDayIncorrect || isDateInFuture || hasSymbols || isOld;
};

export class BirthdayField extends React.Component {
    state = {day: undefined, month: undefined, year: undefined, isDateError: false, allFilled: false};
    componentDidMount() {
        const {value} = this.props;

        if (value && typeof value === 'string') {
            const [year, month, day] = value.split('-');

            this.setBirthdayState({year, month, day, allFilled: true});
        }
    }
    componentDidUpdate(prevProps, prevState) {
        const {setError, isRequired} = this.props;
        const {isDateError, allFilled} = this.state;
        const errorCode = isDateError || (isRequired && !allFilled) ? 'invalid' : '';

        if (prevState.isDateError !== isDateError) {
            setError && setError(errorCode);
        }

        if (prevState.allFilled !== allFilled) {
            setError && setError(errorCode);
        }
    }
    setBirthdayState = ({year, month, day}) => this.setState({year, month, day});
    onChangeDate = (e) => {
        const {onChange, setInProgress, type = 'default'} = this.props;

        if (e && e.target) {
            this.setState({[e.target.name]: e.target.value}, () => {
                const {day, month, year} = this.state;
                const {value} = this.props;
                const allFilled = Boolean(day && month && year);
                const someNotEmpty = Boolean(day || month || year);

                let isValidationError = false;

                this.setState({allFilled});
                if (allFilled) {
                    const dayStr = getDayStr(day);

                    isValidationError = dataValidation({day, month, year, type});
                    onChange(`${year}-${month}-${dayStr}`);
                } else if (value) {
                    onChange('');
                }

                const isInProgress = !allFilled && someNotEmpty;

                setInProgress && setInProgress(isInProgress || isValidationError);
                this.setState({isDateError: !isInProgress && isValidationError});
            });
        }
    };
    onFocus = () => {
        const {onFocus} = this.props;

        onFocus && onFocus();
    };
    render() {
        const {day, month, year} = this.state;
        const {dayPlaceholder, yearPlaceholder, error} = this.props;

        return (
            <div className={b({error: Boolean(error)})} onFocus={this.onFocus}>
                <div className={b('inputWrapper')}>
                    <Input
                        size='l'
                        id='kiddish-birthday-day'
                        name='day'
                        type='text'
                        placeholder={dayPlaceholder}
                        value={day}
                        autoComplete='off'
                        maxLength='2'
                        onChange={this.onChangeDate}
                        pin='round-brick'
                        className={b('dayInput')}
                        state={error ? 'error' : ''}
                    />
                    <Select
                        value={month}
                        id='kiddish-birthday-month'
                        size='l'
                        name='month'
                        pin='clear-clear'
                        onChange={this.onChangeDate}
                        options={MONTHS_OPTIONS}
                    />
                    <Input
                        size='l'
                        id='kiddish-birthday-year'
                        placeholder={yearPlaceholder}
                        name='year'
                        type='text'
                        value={year}
                        autoComplete='off'
                        maxLength='4'
                        onChange={this.onChangeDate}
                        pin='brick-round'
                        className={b('yearInput')}
                        state={error ? 'error' : ''}
                    />
                </div>
            </div>
        );
    }
}

BirthdayField.propTypes = {
    type: PropTypes.oneOf(['default', 'kiddish']),
    onChange: PropTypes.func,
    onFocus: PropTypes.func,
    value: PropTypes.string,
    dayPlaceholder: PropTypes.string,
    yearPlaceholder: PropTypes.string,
    setInProgress: PropTypes.func,
    setError: PropTypes.func,
    error: PropTypes.error,
    isRequired: PropTypes.bool
};
