import React, {Component, PureComponent} from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';
import {Spin} from '@components/Spin';
import {changeCaptchaType, clearCaptchaProps} from './actions';
import reloadCaptcha from './actions/reloadCaptcha';
import stopCaptchaSound from './actions/stopCaptchaSound';
import playCaptchaSound from './actions/playCaptchaSound';
import getCaptcha from './actions/getCaptcha';

class Captcha extends PureComponent {
    constructor(props) {
        super(props);

        this.isAudioCaptchaEnabled = false;
        this.isLoading = true;
    }

    componentDidMount() {
        this.isLoading = false;

        $.ajax({
            url: 'https://yastatic.net/howler/2.1.2/howler.min.js',
            dataType: 'script',
            cache: true,
            scriptAttrs: {
                crossorigin: 'anonymous'
            },
            success: () => {
                this.isAudioCaptchaEnabled = true;
                this.props.dispatch(getCaptcha(true));
            },
            error: () => {
                this.props.dispatch(getCaptcha());
            }
        });

        this.forceUpdate();
    }

    componentDidUpdate() {
        const {loading, imageUrl, captchaKey, dispatch} = this.props;

        const isCaptchaExistWithNullImage = !loading && captchaKey && !imageUrl;

        if (isCaptchaExistWithNullImage) {
            dispatch(reloadCaptcha());
        }
    }

    componentWillUnmount() {
        this.props.dispatch(clearCaptchaProps());
    }

    changeCaptchaType = () => {
        const {type, dispatch} = this.props;

        if (type === 'text') {
            dispatch(playCaptchaSound(true));
        } else {
            dispatch(getCaptcha());
        }

        dispatch(changeCaptchaType());
    };

    reloadCaptcha = () => {
        this.props.dispatch(reloadCaptcha());
    };

    playCaptcha = () => {
        this.props.dispatch(playCaptchaSound());
    };

    render() {
        const {
            imageUrl,
            env,
            captchaKey,
            introSound,
            type,
            loading,
            loadingAudio,
            playing,
            visible,
            lang,
            dispatch
        } = this.props;

        const isShowAudioCaptcha = Boolean(
            (lang === 'ru' || lang === 'tr') && !this.isLoading && this.isAudioCaptchaEnabled
        );

        return (
            <div className={`captcha captcha_type_${type}`}>
                {Boolean(visible) && (
                    <div className='captcha__container'>
                        {type === 'text' && (
                            <CaptchaImage
                                onClick={this.reloadCaptcha}
                                imageUrl={imageUrl}
                                env={env}
                                captchaKey={captchaKey}
                                loading={loading}
                                dispatch={dispatch}
                            />
                        )}

                        {type === 'audio' && isShowAudioCaptcha && (
                            <CaptchaSound
                                loading={loadingAudio}
                                introSound={introSound}
                                playing={playing}
                                dispatch={dispatch}
                                playCaptcha={this.playCaptcha}
                            />
                        )}
                    </div>
                )}

                <div className='captcha__footer clearfix'>
                    {!this.isLoading && (
                        <div className='captcha__reload' onClick={this.reloadCaptcha} role='button' tabIndex='0'>
                            <span className='captcha__reload-icon' />
                            {i18n('_AUTH_.captcha.another-code')}
                        </div>
                    )}

                    {isShowAudioCaptcha && (
                        <div
                            className='captcha__change-type'
                            onClick={this.changeCaptchaType}
                            role='button'
                            tabIndex='0'
                        >
                            <span className='captcha__change-type-icon' />
                            {type === 'text' ? i18n('_AUTH_.captcha.listen-code') : i18n('_AUTH_.captcha.text-code')}
                        </div>
                    )}
                </div>
            </div>
        );
    }
}

Captcha.propTypes = {
    imageUrl: PropTypes.string,
    captchaKey: PropTypes.string,
    env: PropTypes.shape({
        type: PropTypes.string
    }),
    introSound: PropTypes.object,
    captchaSound: PropTypes.object,
    type: PropTypes.string.isRequired,
    loading: PropTypes.bool.isRequired,
    loadingAudio: PropTypes.bool.isRequired,
    playing: PropTypes.bool.isRequired,
    visible: PropTypes.bool.isRequired,
    lang: PropTypes.string.isRequired,
    isMobile: PropTypes.bool,
    dispatch: PropTypes.func.isRequired
};

class CaptchaImage extends Component {
    render() {
        const {loading, imageUrl, captchaKey, env = {}, onClick} = this.props;
        const isNotProduction = env.type === 'development' || env.type === 'testing';

        return loading ? (
            <Spin progress={true} size='l' />
        ) : (
            <img
                src={imageUrl}
                onClick={onClick}
                className='captcha__image'
                alt={isNotProduction ? `captcha image key_${captchaKey}` : 'captcha image'}
                aria-label={isNotProduction ? `captcha image key_${captchaKey}` : 'captcha image'}
            />
        );
    }
}

CaptchaImage.propTypes = {
    onClick: PropTypes.func.isRequired,
    dispatch: PropTypes.func.isRequired,
    imageUrl: PropTypes.string,
    loading: PropTypes.bool.isRequired,
    captchaKey: PropTypes.string,
    env: PropTypes.shape({
        type: PropTypes.string
    })
};

class CaptchaSound extends Component {
    componentWillUnmount() {
        this.props.dispatch(stopCaptchaSound());
    }

    render() {
        const {playing, loading, playCaptcha} = this.props;

        return loading ? (
            <Spin progress={true} size='l' />
        ) : (
            <button
                type='button'
                className={classnames('captcha__audio ', {captcha__audio_playing: playing})}
                onClick={playCaptcha}
                aria-label={i18n('_AUTH_.captcha.listen-code')}
                accessKey='p'
            />
        );
    }
}

CaptchaSound.propTypes = {
    dispatch: PropTypes.func.isRequired,
    playing: PropTypes.bool.isRequired,
    loading: PropTypes.bool.isRequired,
    introSound: PropTypes.object,
    captchaSound: PropTypes.object,
    playCaptcha: PropTypes.func.isRequired
};

export {Captcha};
