import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import {Card} from '@components/Card';
import {Button} from '@plibs/pcomponents/Button';
import metrics from '@blocks/metrics';
import {BILLING_GOAL_PREFIX, BILLING_METRICS_PREFIX} from '@blocks/morda/billing_info';

const b = cn('CardsList');

import './CardsList.styl';

const ERRORS = {
    internal: i18n('Profile.cards.errors.internal'),
    too_many_cards: i18n('Profile.cards.errors.too_many_cards')
};

const MAX_CARDS_COUNT = 5;

export class CardsList extends Component {
    componentDidMount() {
        const {getCardsInfo, billing} = this.props;
        const {cards = []} = billing;

        if (cards.length === 0) {
            getCardsInfo();
        }
    }

    componentWillUnmount() {
        this.props.selectCard('');
    }

    getRealCards = () => {
        const {billing: {cards = []} = {}} = this.props;

        return cards.filter(({proto}) => proto !== 'yamoney_walletapi');
    };

    getErrorsMsgs = (errors = []) => {
        return errors.map((code, index) => <div key={index}>{ERRORS[code] || ERRORS.internal}</div>);
    };

    setFamilyCard = (cardId = '') => {
        const {setFamilyCard} = this.props;

        setFamilyCard(cardId);
    };

    unbindCard = (id) => {
        const {unbindCard} = this.props;

        metrics.send([BILLING_METRICS_PREFIX, 'Отвязать карту']);
        metrics.goal(`${BILLING_GOAL_PREFIX}unbind`);
        unbindCard(id);
    };

    unbindCardOnClick = (id) => {
        this.unbindCard(id);
    };

    createBinding = () => {
        const {createBinding, onBindCard} = this.props;

        if (typeof onBindCard === 'function') {
            onBindCard();
        }

        metrics.send([BILLING_METRICS_PREFIX, 'Привязать карту']);
        metrics.goal(`${BILLING_GOAL_PREFIX}bind_init`);
        createBinding();
    };

    render() {
        const {billing = {}, settings = {}, skipUnbind = true, disableSelect = false} = this.props;
        const {selected = '', errors = [], unbindCardId} = billing;
        const cards = this.getRealCards();
        const {isMobile} = settings.ua;

        return (
            <div className={b('')}>
                {Boolean(errors.length) && (
                    <div className={b('error')}>
                        <div className={b('errorMsg')}>{this.getErrorsMsgs(errors)}</div>
                    </div>
                )}
                {cards.length ? (
                    <ul className={b('container')} data-t='cards-list:list'>
                        {cards.map((card, index) => (
                            <li key={index} className={b('item', {selected: selected === card.id})}>
                                <Card
                                    card={card}
                                    isStatic={true}
                                    onSelect={this.setFamilyCard}
                                    disableSelect={disableSelect}
                                />

                                {(!skipUnbind && card.proto === 'trust') ||
                                    (card.proto === 'fake' && (
                                        <div className={b('unbind')}>
                                            <Button
                                                view='pseudo'
                                                size='l'
                                                type='link'
                                                text={i18n('Profile.cards.unbind')}
                                                disabled={unbindCardId === card.id}
                                                onClick={() => this.unbindCardOnClick(card.id)}
                                                width={isMobile ? 'max' : 'auto'}
                                            />
                                        </div>
                                    ))}
                            </li>
                        ))}
                    </ul>
                ) : (
                    <div className={b('hint')}>{i18n('Profile.cards.info')}</div>
                )}
                {cards.length < MAX_CARDS_COUNT && (
                    <div className={b('add')}>
                        <button type='button' className={b('addButton')} onClick={this.createBinding}>
                            <span className={b('addButtonText')}>{i18n('Profile2.add.bank_card')}</span>
                        </button>
                    </div>
                )}
            </div>
        );
    }
}

CardsList.propTypes = {
    billing: PropTypes.object,
    settings: PropTypes.object,
    retpath: PropTypes.string,
    selectCard: PropTypes.func,
    skipUnbind: PropTypes.bool,
    disableSelect: PropTypes.bool,
    getCardsInfo: PropTypes.func.isRequired,
    unbindCard: PropTypes.func,
    bindCard: PropTypes.func.isRequired,
    setFamilyCard: PropTypes.func,
    createBinding: PropTypes.func.isRequired,
    onBindCard: PropTypes.func,
    onSelect: PropTypes.func
};
