import './CodeField.styl';
import React from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import {Field} from '@components/Field';

const b = cn('CodeField');
const LIMIT_EXCEEDED_ERROR_FRAGMENTS = ['limit_exceeded', 'limit.exceeded', 'limitexceeded'];

class CodeField extends React.PureComponent {
    ref = React.createRef();

    static defaultProps = {
        codeLength: 6
    };

    componentDidMount() {
        if (this.props.skipFocus) {
            return;
        }

        this.ref.current.focus();
    }

    componentDidUpdate(prevProps) {
        if (!prevProps.hasError && this.props.hasError && !this.checkIfLimitExceededError()) {
            this.clearError();
        }
    }

    handleFocus = () => {
        this.ref.current.focus();
    };

    handleChange = (value) => {
        const valueFormatted = value.replace(/[^0-9]+/g, '');

        this.renderNumbers(valueFormatted);
    };

    checkIfLimitExceededError = () => {
        const {error} = this.props;
        const errorCode = error.code;

        if (!errorCode) {
            return false;
        }

        return LIMIT_EXCEEDED_ERROR_FRAGMENTS.some((item) => errorCode.includes(item));
    };

    clearError = () => {
        const {updateErrors, updateValues, isChallegeProcess, setChallengeError, setPhoneConfirmErrors} = this.props;

        setTimeout(() => {
            updateErrors({
                field: 'phoneCode',
                error: {code: '', text: ''}
            });
            updateValues({
                field: 'phoneCode',
                value: ''
            });

            if (isChallegeProcess) {
                setChallengeError({code: '', text: ''});
            }

            setPhoneConfirmErrors([]);
        }, 2300);
    };

    renderNumbers = () => {
        const {codeLength, value} = this.props;
        const list = [];

        if (value) {
            [...value].forEach((item, index) => {
                list.push(
                    <li className={b('number', {hasSpace: codeLength > 4 && index === 3})} key={index}>
                        {item}
                    </li>
                );
            });

            this.renderDashes(list.length - 1);
            return list;
        }

        return null;
    };

    renderDashes = () => {
        const {codeLength, value} = this.props;
        const length = value ? codeLength - value.length : codeLength;
        const checkSpace = (index) => {
            return value.length <= 3 && value.length + index === 2;
        };

        return Array.apply(null, Array(length)).map((item, index) => {
            checkSpace(value, index);
            return (
                <li
                    className={b('dash', {isActive: index === 0, hasSpace: codeLength > 4 && checkSpace(index)})}
                    key={index}
                />
            );
        });
    };

    renderPseudoInput = () => {
        const {value, hasError, codeLength} = this.props;
        const isCodeLengthOk = value.length === codeLength;

        return (
            <div className={b('visualWrapper')} aria-hidden='true'>
                <div className={b('visualContent', {size: codeLength === 4 ? 'small' : 'normal'})}>
                    <ul
                        className={b('dashes', {
                            hasError: hasError && isCodeLengthOk,
                            isCodeIncomplete: hasError && !isCodeLengthOk
                        })}
                    >
                        {this.renderDashes()}
                    </ul>
                    <ul className={b('numbers', {hasError: hasError && isCodeLengthOk})}>{this.renderNumbers()}</ul>
                </div>
            </div>
        );
    };

    renderCallPseudoInput = () => {
        const {value, hasError, codeLength, callingNumberTemplate} = this.props;
        const isCodeLengthOk = value.length === codeLength;

        return (
            <div className={b('visualWrapper')} aria-hidden='true'>
                <span className={b('callingNumber')}>{callingNumberTemplate.replace('-XX-XX', '')}</span>
                <div className={b('visualContent', {size: 'small'})}>
                    <ul
                        className={b('dashes', {
                            hasError: hasError && isCodeLengthOk,
                            isCodeIncomplete: hasError && !isCodeLengthOk
                        })}
                    >
                        {this.renderDashes()}
                    </ul>
                    <ul className={b('numbers', {hasError: hasError && isCodeLengthOk})}>{this.renderNumbers()}</ul>
                </div>
            </div>
        );
    };

    render() {
        const {name, codeLength, callingNumberTemplate, error, onKeyUp} = this.props;
        const isDisabled = this.checkIfLimitExceededError();
        const {code: errorCode} = error;

        return (
            <div className={b('', {disabled: isDisabled})} onClick={this.handleFocus}>
                <Field
                    onChange={this.handleChange}
                    name={name || 'phoneCode'}
                    size='xxl'
                    ref={this.ref}
                    error={error}
                    disabled={isDisabled}
                    view='rounded-corner'
                    options={{maxLength: callingNumberTemplate ? 4 : codeLength, inputMode: 'numeric'}}
                    isPhoneCode={true}
                    onKeyUp={onKeyUp}
                    hideLabel={true}
                    hideError={['code.invalid', 'code.empty'].includes(errorCode)}
                />

                {callingNumberTemplate ? this.renderCallPseudoInput() : this.renderPseudoInput()}
            </div>
        );
    }
}

CodeField.propTypes = {
    name: PropTypes.string,
    value: PropTypes.string,
    skipFocus: PropTypes.bool,
    hasError: PropTypes.bool,
    isChallegeProcess: PropTypes.bool,
    updateErrors: PropTypes.func.isRequired,
    updateValues: PropTypes.func.isRequired,
    onKeyUp: PropTypes.func,
    setChallengeError: PropTypes.func,
    setPhoneConfirmErrors: PropTypes.func,
    codeLength: PropTypes.number,
    error: PropTypes.object,
    callingNumberTemplate: PropTypes.string
};

export {CodeField};
