import './Field.styl';
import React from 'react';
import PropTypes from 'prop-types';
import debounce from 'lodash/debounce';
import {cn} from '@bem-react/classname';
import {Input} from '@components/Input';
import {Link} from '@components/Link';
import {NATIVE_INPUT_ID_PREFIX} from '@components/Field/names';
import {getError, focusInput, checkDigitsAndDash} from '@blocks/utils';
import metrika from '@blocks/metrics';
import {createPhoneMask} from './validations';

import MaskedInput from 'react-text-mask';
import classnames from 'classnames';

const b = cn('Field');

const INPUT_PHONE_PLACEHOLDER = '+7 (000) 000-00-00';

class Field extends React.PureComponent {
    static defaultProps = {
        type: 'text'
    };

    state = {
        isFocused: false
    };

    ref = React.createRef();

    focus = () => focusInput(this.field);

    onKeyUp = () => {
        const {onKeyUp} = this.props;

        if (typeof onKeyUp === 'function') {
            onKeyUp();
        }
    };

    onChange = (event) => {
        const {target = {}} = event;
        const {value} = target;
        const {name, updateValues, onChange, isValueForced, isPhoneCode} = this.props;

        if (!isPhoneCode && (!isValueForced || !onChange)) {
            // Не берем из стора и не пишем в него если value явно передается снаружи
            // https://st.yandex-team.ru/PASSP-28422
            updateValues({field: name, value});
        }

        if (onChange && typeof onChange === 'function') {
            onChange(value);
        }

        if (isPhoneCode) {
            this.handleDigitValue(value);
        }
    };

    onFieldChange = (value = '') => {
        const {name, updateValues, onChange, isValueForced} = this.props;

        if (!isValueForced || !onChange) {
            updateValues({field: name, value});
        }

        if (onChange && typeof onChange === 'function') {
            onChange(value);
        }
    };

    handleDigitValue = (value) => {
        const {name, updateValues, hasRoundViewExp} = this.props;
        const isValid = checkDigitsAndDash(value);

        if (hasRoundViewExp) {
            return updateValues({field: name, value: value.replace(/[^0-9]+/g, '')});
        }

        if (isValid) {
            updateValues({field: name, value});
        }
    };

    componentDidUpdate(prevProps) {
        const {value, isErrorForced} = this.props;

        if (value !== prevProps.value && !isErrorForced) {
            this.validate();
        }
    }

    _validate = () => {
        const {
            name,
            value,
            updateErrors,
            validate,
            setPasswordWeight,
            shouldValidate,
            isErrorForced,
            validationTrack
        } = this.props;

        // Если ошбика явно зафоршена снаружи, то валидация и запись в стор не требуются
        // https://st.yandex-team.ru/PASSP-28422
        if (isErrorForced) {
            return;
        }

        if (!shouldValidate) {
            return updateErrors({field: name, error: {}});
        }

        validate(name, {validationTrack}).then((response = {}) => {
            const {code, status} = response;

            if (code) {
                metrika.send(`Показ ошибки поля ${name}: ${code}`);
            }

            if (name === 'password') {
                if ((value && !code) || ['weak', 'tooshort'].includes(code)) {
                    setPasswordWeight(code || 'strong');

                    if (status === 'warning') {
                        updateErrors({field: name, error: code ? getError(name, code, true) : {}});
                        return;
                    }
                } else {
                    setPasswordWeight('');
                }
            }

            updateErrors({field: name, error: status === 'error' && code ? getError(name, code) : {}});
        });
    };

    validate = debounce(this._validate, 100);

    onFocus = () => this.setState({isFocused: true});
    onBlur = () => {
        const {onBlur} = this.props;

        this.setState({isFocused: false});

        if (typeof onBlur === 'function') {
            onBlur();
        }
    };

    renderPhoneField = () => {
        const {name, value, error, isValid, shrinkBottomMargin, validHint, tInfo, isAm} = this.props;
        const {isFocused} = this.state;
        const errorMessage = typeof error === 'object' && error !== null ? error.text : error;
        const errorDescription = typeof error === 'object' && error !== null ? error.errorDescription : '';
        const errorCode = typeof error === 'object' && error !== null ? error.code : '';

        const state = {
            state: errorMessage ? 'error' : isValid ? 'success' : undefined,
            hint: errorMessage ? errorMessage : isValid && validHint ? validHint : '',
            description: errorDescription
        };

        if (typeof error === 'object' && error !== null && error.warning) {
            state.state = 'warning';
        }

        return (
            <div
                aria-label={tInfo}
                className={b({
                    view: 'rounded-corner',
                    shrinkBottomMargin,
                    isFocused,
                    isFilled: Boolean(value),
                    isAm
                })}
                data-t={`field:${name}`}
                data-t-error={errorCode ? `field:error-${name}:${errorCode}` : undefined}
            >
                {this.maybeRenderLabel()}
                {this.maybeRenderErrorIcon(true)}
                <span
                    className={classnames(
                        'Textinput Textinput_view_rounded-corner Textinput_size_l Textinput_phone-mask',
                        {
                            Textinput_state_error: errorCode,
                            Textinput_focused: isFocused
                        }
                    )}
                >
                    <MaskedInput
                        mask={createPhoneMask(value)}
                        placeholder={INPUT_PHONE_PLACEHOLDER}
                        className='Textinput-Control Textinput-Control_phone'
                        guide={false}
                        id='passp-field-phone'
                        onBlur={this.onBlur}
                        onFocus={this.onFocus}
                        onChange={this.onChange}
                        inputMode='tel'
                        value={value}
                    />
                    <span className='Textinput-Box' />
                </span>
                {errorMessage && (
                    <div
                        id='field:input-phone:hint'
                        role='alert'
                        aria-live='assertive'
                        data-t='field:input-phone:hint'
                        className='Textinput-Hint Textinput-Hint_state_error'
                    >
                        {errorMessage}
                    </div>
                )}
            </div>
        );
    };

    maybeRenderLink = () => {
        const {name, link, isShowFieldLink} = this.props;
        const elemId = `field:link-${name}`;

        if (!isShowFieldLink || !link) {
            return false;
        }

        return (
            <div className={b('link')} data-t={elemId} id={elemId}>
                <Link href={link.url} onClick={link.onClick} pseudo={link.pseudo || false} weight='medium'>
                    {link.text}
                </Link>
            </div>
        );
    };

    maybeRenderLabel = () => {
        const {label, name, hideLabel} = this.props;

        if (!label || hideLabel) {
            return null;
        }

        return (
            <div className={b('label-wrapper')}>
                <label
                    aria-hidden={Boolean(name).toString()}
                    className={b('label')}
                    data-t={`field:label-${name}`}
                    htmlFor={`${NATIVE_INPUT_ID_PREFIX}${name}`}
                >
                    {label}
                </label>
            </div>
        );
    };

    maybeRenderUnvoicedPlaceholder = () => {
        const {value, unvoicedPlaceholder, placeholder, label, hideLabel} = this.props;

        if (value || !unvoicedPlaceholder || placeholder) {
            return null;
        }

        return (
            <div className={b('unvoiced-placeholder', {hasLabel: label && !hideLabel})} aria-hidden='true'>
                {unvoicedPlaceholder}
            </div>
        );
    };

    maybeRenderErrorIcon = (showErrorIcon) => {
        const {error, hasErrorIcon} = this.props;
        const {isFocused} = this.state;
        const errorCode = typeof error === 'object' && error !== null ? error.code : error;

        if (!isFocused && errorCode && (showErrorIcon || hasErrorIcon)) {
            return <span className={b('errorIcon')} aria-hidden={true} />;
        }
    };

    render() {
        const {
            size,
            name,
            value,
            error,
            type,
            options,
            placeholder,
            isValid,
            iconRight,
            view,
            disabled,
            maxLength,
            shrinkBottomMargin,
            validHint,
            tInfo,
            isAm,
            dir,
            hasRoundViewExp,
            hasPhoneMask,
            isPhoneCode,
            hideError
        } = this.props;
        const {isFocused} = this.state;
        const errorMessage = typeof error === 'object' && error !== null ? error.text : error;
        const errorDescription = typeof error === 'object' && error !== null ? error.errorDescription : '';
        const errorCode = typeof error === 'object' && error !== null ? error.code : '';

        const state = {
            state: errorMessage ? 'error' : isValid ? 'success' : undefined,
            hint: errorMessage && !hideError ? errorMessage : isValid && validHint ? validHint : '',
            description: errorDescription
        };

        if (typeof error === 'object' && error !== null && error.warning) {
            state.state = 'warning';
        }

        if (hasPhoneMask) {
            return this.renderPhoneField();
        }

        return (
            <div
                aria-label={tInfo}
                className={b({
                    view: hasRoundViewExp ? 'rounded-corner' : view,
                    shrinkBottomMargin,
                    isFocused,
                    isFilled: Boolean(value),
                    isAm
                })}
                data-t={`field:${name}`}
                data-t-error={errorCode ? `field:error-${name}:${errorCode}` : undefined}
            >
                {this.maybeRenderLabel()}
                {this.maybeRenderErrorIcon(hasRoundViewExp && !isPhoneCode)}
                <Input
                    controlRef={(ref) => (this.field = ref)}
                    onChange={this.onChange}
                    onFocus={this.onFocus}
                    onBlur={this.onBlur}
                    onKeyUp={this.onKeyUp}
                    name={name}
                    placeholder={view === 'floating-label' ? '' : placeholder}
                    value={value}
                    type={type}
                    size={size}
                    dir={dir}
                    id={`${NATIVE_INPUT_ID_PREFIX}${name}`}
                    data-t={`field:input-${name}`}
                    aria-label={tInfo}
                    iconRight={iconRight}
                    view={hasRoundViewExp ? 'rounded-corner' : view}
                    disabled={disabled}
                    maxLength={maxLength}
                    {...state}
                    {...options}
                />
                {this.maybeRenderUnvoicedPlaceholder()}
                {this.maybeRenderLink()}
            </div>
        );
    }
}

Field.propTypes = {
    size: PropTypes.oneOf(['s', 'm', 'l', 'xxl']),
    passwordWeight: PropTypes.string,
    type: PropTypes.string,
    name: PropTypes.string.isRequired,
    label: PropTypes.string,
    value: PropTypes.oneOfType([PropTypes.string, PropTypes.bool]),
    error: PropTypes.oneOfType([
        PropTypes.string,
        PropTypes.shape({
            text: PropTypes.string,
            code: PropTypes.string
        })
    ]),
    updateValues: PropTypes.func.isRequired,
    updateErrors: PropTypes.func.isRequired,
    setPasswordWeight: PropTypes.func,
    onChange: PropTypes.func,
    onKeyUp: PropTypes.func,
    validate: PropTypes.func,
    options: PropTypes.shape({
        autoCorrect: PropTypes.string,
        autoCapitalize: PropTypes.string,
        autoComplete: PropTypes.string
    }),
    placeholder: PropTypes.string,
    className: PropTypes.string,
    link: PropTypes.shape({
        url: PropTypes.string,
        text: PropTypes.string,
        onClick: PropTypes.func,
        pseudo: PropTypes.bool
    }),
    isShowFieldLink: PropTypes.bool,
    isValid: PropTypes.bool,
    validHint: PropTypes.string,
    shouldValidate: PropTypes.bool,
    isValueForced: PropTypes.bool,
    isErrorForced: PropTypes.bool,
    hasErrorIcon: PropTypes.bool,
    hideError: PropTypes.bool,
    onBlur: PropTypes.func,
    iconRight: PropTypes.node,
    view: PropTypes.oneOf(['default', 'floating-label', 'one-border', 'big-input', 'rounded-corner']),
    disabled: PropTypes.bool,
    maxLength: PropTypes.number,
    shrinkBottomMargin: PropTypes.bool,
    hideLabel: PropTypes.bool,
    validationTrack: PropTypes.string,
    tInfo: PropTypes.string,
    isAm: PropTypes.bool,
    isPhoneCode: PropTypes.bool,
    dir: PropTypes.string,
    hasRoundViewExp: PropTypes.bool,
    unvoicedPlaceholder: PropTypes.string,
    hasPhoneMask: PropTypes.bool
};

export {Field};
