import api from '@blocks/api';
import {getAnyTypeTrackId} from '@blocks/selectors';

const PHONE_FIELD_MASKS = {
    russia_local: [
        /[1-9]/,
        ' ',
        '(',
        /\d/,
        /\d/,
        /\d/,
        ')',
        ' ',
        /\d/,
        /\d/,
        /\d/,
        '-',
        /\d/,
        /\d/,
        '-',
        /\d/,
        /\d/,
        /\d/,
        /\d/,
        /\d/,
        /\d/
    ],
    russia: [
        '+',
        /[1-9]/,
        ' ',
        '(',
        /\d/,
        /\d/,
        /\d/,
        ')',
        ' ',
        /\d/,
        /\d/,
        /\d/,
        '-',
        /\d/,
        /\d/,
        '-',
        /\d/,
        /\d/,
        /\d/,
        /\d/,
        /\d/,
        /\d/
    ],
    universal: ['+', /\d/, /\d/, /\d/, /\d/, /\d/, /\d/, /\d/, /\d/, /\d/, /\d/, /\d/, /\d/, /\d/, /\d/, /\d/],
    short: [
        '+',
        /[1-9]/,
        ' ',
        '(',
        /\d/,
        /\d/,
        /\d/,
        ')',
        ' ',
        /\d/,
        /\d/,
        /\d/,
        '-',
        /\d/,
        /\d/,
        /\d/,
        /\d/,
        /\d/,
        /\d/,
        /\d/,
        /\d/
    ],
    middle: [
        '+',
        /[1-9]/,
        /\d/,
        ' ',
        '(',
        /\d/,
        /\d/,
        /\d/,
        ')',
        ' ',
        /\d/,
        /\d/,
        /\d/,
        ' ',
        /\d/,
        /\d/,
        ' ',
        /\d/,
        /\d/,
        /\d/,
        /\d/,
        /\d/
    ],
    long: [
        '+',
        /[1-9]/,
        /\d/,
        /\d/,
        ' ',
        '(',
        /\d/,
        /\d/,
        ')',
        ' ',
        /\d/,
        /\d/,
        /\d/,
        ' ',
        /\d/,
        /\d/,
        /\d/,
        /\d/,
        /\d/,
        /\d/,
        /\d/
    ],
    uk: ['+', /[1-9]/, /\d/, ' ', /\d/, /\d/, /\d/, /\d/, ' ', /\d/, /\d/, /\d/, /\d/, /\d/, /\d/, /\d/, /\d/],
    netherlands: [
        '+',
        /[1-9]/,
        /\d/,
        ' ',
        /\d/,
        ' ',
        /\d/,
        /\d/,
        ' ',
        /\d/,
        /\d/,
        /\d/,
        ' ',
        /\d/,
        /\d/,
        /\d/,
        /\d/,
        /\d/,
        /\d/
    ],
    swiss: [
        '+',
        /[1-9]/,
        /\d/,
        ' ',
        /\d/,
        /\d/,
        ' ',
        /\d/,
        /\d/,
        /\d/,
        '-',
        /\d/,
        /\d/,
        '-',
        /\d/,
        /\d/,
        /\d/,
        /\d/,
        /\d/
    ],
    angola: ['+', /[1-9]/, /\d/, /\d/, ' ', /\d/, /\d/, /\d/, ' ', /\d/, /\d/, /\d/, ' ', /\d/, /\d/, /\d/, /\d/, /\d/]
};

class Validations {
    _ok = {status: 'ok'};
    _error = (code) => ({status: 'error', code});
    _warning = (code) => ({status: 'warning', code});

    _getDefaultRequestParams = (state = {}, validationTrack) => {
        const {common = {}} = state;
        const {csrf} = common;
        const trackId = getAnyTypeTrackId(state);

        return {
            csrf_token: csrf,
            track_id: validationTrack || trackId
        };
    };

    _getError = (validation = {}) => {
        const errors = validation.validation_errors || validation.errors;
        const error = (errors && errors[0]) || (Array.isArray(validation) && {code: validation[0]});
        const {field, code, message = ''} = error;

        if (field === 'password' && message.indexOf('10 symbols') !== -1 && code === 'tooshort') {
            return 'strong_pwd.tooshort';
        }

        return error ? error.code || error : undefined;
    };

    _getWarning = (validation = {}) => {
        const warnings = validation.validation_warnings;
        const warning = warnings && warnings[0];

        return warning ? warning.code || warning : undefined;
    };

    _request = (method = '', params = {}, resolve) =>
        api
            .request(method, params)
            .done((validation = {}) => {
                const error = this._getError(validation);
                const warning = this._getWarning(validation);

                if (warning) {
                    return resolve(this._warning(warning));
                }

                if (error) {
                    return resolve(this._error(error));
                }

                return resolve(this._ok);
            })
            .fail((validation = {}) => {
                const error = this._getError(validation);

                if (error) {
                    return resolve(this._error(error));
                }

                return resolve(this._ok);
            });

    login = (state = {}) => {
        const {form = {}} = state;
        const {values = {}} = form;
        const {login} = values;

        return new Promise((resolve) => {
            if (login) {
                const params = Object.assign({}, this._getDefaultRequestParams(state), {login});

                return this._request('login', params, resolve);
            }

            return resolve(this._ok);
        });
    };

    password = (state = {}, {validationTrack}) => {
        const {form = {}} = state;
        const {values = {}} = form;
        const {phone: phone_number, login, password} = values;

        return new Promise((resolve) => {
            if (password) {
                const params = Object.assign({}, this._getDefaultRequestParams(state, validationTrack), {
                    password,
                    login,
                    phone_number
                });

                return this._request('password', params, resolve);
            }

            return resolve(this._ok);
        });
    };

    password_confirm = (state = {}) => {
        const {form = {}} = state;
        const {values = {}} = form;
        const {password, password_confirm: passwordConfirm} = values;

        return new Promise((resolve) => {
            if (password && passwordConfirm && password !== passwordConfirm) {
                return resolve(this._error('notequal'));
            }

            return resolve(this._ok);
        });
    };

    firstname = (state = {}) => {
        const {form = {}} = state;
        const {values = {}} = form;
        const {firstname} = values;

        return new Promise((resolve) => {
            if (firstname) {
                const params = Object.assign({}, this._getDefaultRequestParams(state), {firstname});

                return this._request('validate-firstname', params, resolve);
            }

            return resolve(this._ok);
        });
    };

    lastname = (state = {}) => {
        const {form = {}} = state;
        const {values = {}} = form;
        const {lastname} = values;

        return new Promise((resolve) => {
            if (lastname) {
                const params = Object.assign({}, this._getDefaultRequestParams(state), {lastname});

                return this._request('validate-lastname', params, resolve);
            }

            return resolve(this._ok);
        });
    };
}

const validations = new Validations();

export const validate = (field = '', options) => (dispatch, getState) =>
    (validations[field] && validations[field](getState(), options)) || Promise.resolve();

export function createPhoneMask(value) {
    const clearValue = value.replace(/[^0-9]+/g, '');
    const valueArr = clearValue.split('');

    if (valueArr.length === 0) {
        return PHONE_FIELD_MASKS.russia;
    }

    if (valueArr.length === 1 && valueArr[0] === '8') {
        return PHONE_FIELD_MASKS.russia_local;
    }

    if (valueArr.length > 1) {
        switch (valueArr[0]) {
            case '1':
                return PHONE_FIELD_MASKS.short;
            case '2':
                switch (valueArr[1]) {
                    case '0':
                    case '7':
                        return PHONE_FIELD_MASKS.middle;
                    case '4':
                        switch (valueArr[2]) {
                            case '4':
                                return PHONE_FIELD_MASKS.angola;
                            default:
                                return PHONE_FIELD_MASKS.long;
                        }
                    default:
                        return PHONE_FIELD_MASKS.long;
                }
            case '3':
                switch (valueArr[1]) {
                    case '1':
                        return PHONE_FIELD_MASKS.netherlands;
                    case '5':
                    case '7':
                    case '8':
                        return PHONE_FIELD_MASKS.long;
                    default:
                        return PHONE_FIELD_MASKS.middle;
                }
            case '4':
                switch (valueArr[1]) {
                    case '1':
                        return PHONE_FIELD_MASKS.swiss;
                    case '2':
                        return PHONE_FIELD_MASKS.long;
                    case '4':
                        return PHONE_FIELD_MASKS.uk;
                    default:
                        return PHONE_FIELD_MASKS.middle;
                }
            case '5':
                switch (valueArr[1]) {
                    case '0':
                    case '9':
                        return PHONE_FIELD_MASKS.long;
                    default:
                        return PHONE_FIELD_MASKS.middle;
                }
            case '6':
                switch (valueArr[1]) {
                    case '7':
                    case '8':
                    case '9':
                        return PHONE_FIELD_MASKS.long;
                    default:
                        return PHONE_FIELD_MASKS.middle;
                }
            case '7':
                switch (valueArr[1]) {
                    default:
                        return PHONE_FIELD_MASKS.russia;
                }
            case '8':
                switch (valueArr[1]) {
                    case '9':
                        return PHONE_FIELD_MASKS.russia_local;
                    case '5':
                    case '8':
                        return PHONE_FIELD_MASKS.long;
                    default:
                        return PHONE_FIELD_MASKS.middle;
                }
            case '9':
                switch (valueArr[1]) {
                    case '6':
                    case '7':
                    case '9':
                        return PHONE_FIELD_MASKS.long;
                    default:
                        return PHONE_FIELD_MASKS.middle;
                }
            default:
                return PHONE_FIELD_MASKS.universal;
        }
    }

    return PHONE_FIELD_MASKS.universal;
}
