import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import {cn} from '@bem-react/classname';
import {Popup} from '@components/Popup';
import {Menu} from '@components/Menu';
import {Link} from '@components/Link';
import metrics from '@blocks/metrics';
import {links, metricsData} from './common';
import './Footer.styl';

const b = cn('Footer');

export class Footer extends Component {
    langswitcherAnchor = React.createRef();

    static mapStateToProps(state) {
        const {
            settings: {tld, env, help, language},
            common: {isWebView, loadCustomsStyles},
            footer: {langlist}
        } = state;

        return {
            langlist,
            tld,
            env,
            help,
            isWebView,
            language,
            withCustomsStyles: loadCustomsStyles
        };
    }

    constructor(props) {
        super(props);

        this.openLangswitcherPopup = this.openLangswitcherPopup.bind(this);
        this.onOutsideClick = this.onOutsideClick.bind(this);
        this.getItems = this.getItems.bind(this);
        this.state = {
            langPopupOpen: false
        };

        this.isShowHelpLink = !props.env || props.env.name !== 'intranet';
    }

    getItems() {
        if (this.props.env && this.props.env.name === 'intranet') {
            return [];
        }

        if (!this.props.langlist) {
            return [];
        }

        return this.props.langlist
            .filter((item) => item.id !== this.props.language)
            .map((item) => ({
                content: item.name,
                value: item.link
            }));
    }

    openLangswitcherPopup() {
        this.setState({
            langPopupOpen: true
        });
    }

    onOutsideClick() {
        this.setState({
            langPopupOpen: false
        });
    }

    onLangChange = (event = {}) => {
        const {target = {}} = event;
        const {value} = target;

        if (typeof window !== 'undefined') {
            location.href = value;
        }
    };

    render() {
        const {
            tld,
            page,
            helpUrl,
            language,
            env = {},
            help = {},
            isWebView,
            withCustomsStyles,
            isFeedbackLinkHidden
        } = this.props;
        const currentYear = new Date().getFullYear();
        const options = this.getItems();
        const isIntranet = env.name === 'intranet';
        const hasFeedbackLink = !isFeedbackLinkHidden && !isIntranet && tld === 'ru';

        let helpLinkTld = tld;

        if (['az', 'com.am', 'com.ge', 'kg', 'md', 'tj', 'tm', 'uz'].indexOf(tld) > -1) {
            helpLinkTld = 'ru';
        }

        if (['co.il', 'fr', 'lv', 'lt', 'ee', 'fi', 'pl', 'eu'].indexOf(tld) > -1) {
            helpLinkTld = 'com';
        }

        const helpLink = helpUrl || (help.passport && help.passport.replace('%tld%', helpLinkTld)) || '#';
        const additionalClass = page ? `footer-${page}` : '';

        return (
            <div className={`${b({withCustomsStyles})}${additionalClass ? ` ${additionalClass}` : ''}`}>
                {Boolean(tld !== 'com' && !isIntranet && options.length) && (
                    <span>
                        <span
                            className={b('item', {langswitcher: true})}
                            onMouseDown={() => metrics.send(metricsData.langSwitcher)}
                        >
                            <Link
                                pseudo={true}
                                onClick={this.openLangswitcherPopup}
                                controlRef={this.langswitcherAnchor}
                            >
                                {language}
                            </Link>
                        </span>
                        <Popup
                            target='anchor'
                            visible={this.state.langPopupOpen}
                            anchor={this.langswitcherAnchor}
                            directions={['top-center']}
                            onClose={this.onOutsideClick}
                        >
                            <Menu size='s' width='max' items={options} onChange={this.onLangChange} />
                        </Popup>
                    </span>
                )}

                {!isWebView && this.isShowHelpLink && (
                    <span className={b('item')} onMouseDown={() => metrics.send(metricsData.help)}>
                        <Link href={helpLink} target='_blank'>
                            {i18n('Frontend.footer.reference')}
                        </Link>
                    </span>
                )}

                {hasFeedbackLink && (
                    <span className={b('item')} onMouseDown={() => metrics.send(metricsData.feedback)}>
                        <Link href={links.feedback} target='_blank'>
                            {i18n('Frontend.footer.feedback')}
                        </Link>
                    </span>
                )}

                <span className={b('item', {rights: true})}>
                    {`© ${tld === 'ru' ? '2001-' : ''}${currentYear}, `}
                    {!isWebView ? <Link href={`//yandex.${tld}`}>{i18n('_AUTH_.yandex')}</Link> : i18n('_AUTH_.yandex')}
                </span>
            </div>
        );
    }
}

Footer.propTypes = {
    children: PropTypes.oneOfType([PropTypes.array, PropTypes.element]),
    env: PropTypes.object.isRequired,
    help: PropTypes.object.isRequired,
    helpUrl: PropTypes.string,
    isWebView: PropTypes.bool,
    isFeedbackLinkHidden: PropTypes.bool,
    langlist: PropTypes.array.isRequired,
    language: PropTypes.string.isRequired,
    page: PropTypes.string,
    tld: PropTypes.string.isRequired,
    withCustomsStyles: PropTypes.bool
};

export default connect(Footer.mapStateToProps)(Footer);
