import React, {Children} from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';

import {Popup} from '@components/Popup';
import {Menu} from '@components/Menu';
import {Link} from '@components/Link';

import metrics from '@blocks/metrics';
import {metricsData} from './common';

const EULA_LINK = 'https://yandex.ru/legal/yandex_plus_conditions/';

export default class Footer extends React.Component {
    langswitcherAnchor = React.createRef();

    constructor(props) {
        super(props);

        let tld = props.tld;

        if (['az', 'com.am', 'com.ge', 'kg', 'md', 'tj', 'tm', 'uz'].indexOf(tld) > -1) {
            tld = 'ru';
        } else if (['co.il', 'fr', 'lv', 'lt', 'ee', 'fi', 'pl', 'eu'].indexOf(tld) > -1) {
            tld = 'com';
        }

        this.calcCanFooterBeFixed = this.calcCanFooterBeFixed.bind(this);
        this.updateWindowHeight = this.updateWindowHeight.bind(this);
        this.openLangswitcherPopup = this.openLangswitcherPopup.bind(this);
        this.onOutsideClick = this.onOutsideClick.bind(this);
        this.getItems = this.getItems.bind(this);
        this.topCenter = ['top-center'];
        this.state = {
            langPopupOpen: false,
            windowHeight: 0,
            canFooterBeFixed: false
        };
        this.metrics = {
            langSwitcher: () => {
                metrics.send(metricsData.langSwitcher);
            },
            help: () => {
                metrics.send(metricsData.help);
            },
            feedback: () => {
                metrics.send(metricsData.feedback);
            },
            eula: () => {
                metrics.send(metricsData.eula);
            }
        };
        this.currentYear = new Date().getFullYear();
        this.options = this.getItems();
        this.helpLink = ((props.help || {}).passport || '#').replace('%tld%', tld);
    }

    componentDidMount() {
        window.addEventListener('resize', this.updateWindowHeight);
        this.calcCanFooterBeFixed();
    }

    componentWillUnmount() {
        window.removeEventListener('resize', this.updateWindowHeight);
    }

    componentDidUpdate(prevProps, prevState) {
        if (this.state.windowHeight !== prevState.windowHeight) {
            this.calcCanFooterBeFixed();
        }
    }

    updateWindowHeight() {
        this.setState({windowHeight: window.innerHeight});
    }

    getItems() {
        const {env = {}, langlist, language} = this.props;

        if (!langlist || env.name === 'intranet') {
            return [];
        }

        return langlist
            .filter((item) => item.id !== language)
            .map((item) => ({
                content: item.name,
                value: item.link
            }));
    }

    onLangChange = (event = {}) => {
        const {target = {}} = event;
        const {value} = target;

        if (typeof window !== 'undefined') {
            location.href = value;
        }
    };

    openLangswitcherPopup() {
        this.setState({
            langPopupOpen: true
        });
    }

    onOutsideClick() {
        this.setState({
            langPopupOpen: false
        });
    }

    calcCanFooterBeFixed() {
        const asideNavBlock = document.querySelector('.PageNavigation_aside');

        if (!asideNavBlock) {
            return true;
        }

        const distanceToBottom = window.innerHeight - asideNavBlock.offsetTop - asideNavBlock.offsetHeight;

        this.setState({canFooterBeFixed: distanceToBottom >= 140});
    }

    render() {
        const {env = {}, tld, page = '', language, children, location = '', isMobile} = this.props;
        const {langPopupOpen, canFooterBeFixed} = this.state;
        const isIntranet = env.name === 'intranet';
        const additionalClass = page.length > 0 ? `footer-${page}` : '';
        const hasSwitcher = tld !== 'com' && !isIntranet && this.options.length > 0;
        const isFamilyPage = location.startsWith('/profile/family');
        const isSecondRow = [
            '/profile/family',
            '/support',
            '/profile/passwords',
            '/profile/bonus',
            '/profile/social'
        ].some((path) => location.startsWith(path));
        const hasFeedback = tld === 'ru' && !isIntranet;

        return (
            <div
                className={classnames('n-footer', additionalClass, {
                    'n-footer_size_xl': hasFeedback && isFamilyPage,
                    'n-footer_size_l': hasFeedback || isFamilyPage,
                    'n-footer_size_m': !hasFeedback && !isFamilyPage,
                    'n-footer_isFamily': isFamilyPage,
                    'n-footer_isSecondRow': isSecondRow
                })}
            >
                <div className={classnames('n-footer__inner', {'n-footer__inner_notFixed': !canFooterBeFixed})}>
                    <div className='n-footer__body'>
                        {((children && Children.toArray(children).length > 0) || isFamilyPage) && (
                            <div className='n-footer__children'>
                                {children}
                                {isFamilyPage && (
                                    <span className='n-footer__children-item' onClick={this.metrics.eula}>
                                        <Link href={EULA_LINK} target='_blank' rel='noopener'>
                                            {i18n('Plus.p2.plus.footer.license')}
                                        </Link>
                                    </span>
                                )}
                            </div>
                        )}
                        <div className='n-footer__info'>
                            {hasSwitcher && (
                                <span
                                    className='n-footer__item n-footer__langswitcher'
                                    onClick={this.metrics.langSwitcher}
                                >
                                    <Link
                                        pseudo={true}
                                        onClick={this.openLangswitcherPopup}
                                        controlRef={this.langswitcherAnchor}
                                    >
                                        {language}
                                    </Link>
                                </span>
                            )}
                            {isMobile && (
                                <span className='n-footer__item' onClick={this.metrics.help}>
                                    <Link href={this.helpLink} target='_blank' rel='noopener'>
                                        {i18n('Frontend.footer.reference')}
                                    </Link>
                                </span>
                            )}
                            {hasSwitcher && (
                                <Popup
                                    target='anchor'
                                    visible={langPopupOpen}
                                    anchor={this.langswitcherAnchor}
                                    directions={this.topCenter}
                                    onClose={this.onOutsideClick}
                                >
                                    <Menu size='s' width='max' items={this.options} onChange={this.onLangChange} />
                                </Popup>
                            )}
                        </div>
                        <div className='n-footer__rights'>
                            {`© ${tld === 'ru' ? '2001-' : ''}${this.currentYear}, `}
                            <Link href={`//yandex.${tld}`}>{i18n('_AUTH_.yandex')}</Link>
                        </div>
                    </div>
                </div>
            </div>
        );
    }
}

Footer.propTypes = {
    isMobile: PropTypes.bool.isRequired,
    settings: PropTypes.object.isRequired,
    env: PropTypes.object.isRequired,
    help: PropTypes.object.isRequired,
    langlist: PropTypes.array.isRequired,
    page: PropTypes.string,
    location: PropTypes.string,
    tld: PropTypes.string.isRequired,
    language: PropTypes.string.isRequired,
    children: PropTypes.oneOfType([PropTypes.array, PropTypes.element])
};
