import './Header.styl';
import React from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';
import {connect} from 'react-redux';
import {Link as RouterLink} from 'react-router-dom';
import {Logo} from '@components/Logo';
import {Button} from '@components/Button';
import {User} from '@components/User';
import {sendRumRaf, RUM_MARKS} from '@plibs/rum-counter/utils';
import {hasExp} from '@blocks/selectors';
import {UserIcon} from './icons';

const FallbackLink = (props) => (
    <a
        {...Object.assign({}, props, {
            to: undefined,
            href: props.to
        })}
    />
);

FallbackLink.propTypes = {
    to: PropTypes.string
};

const isLinkActive = (link = {}, pathname = '') => {
    const href = link.href || link.to;

    return (href && pathname.includes(href)) || (!pathname && href === '/profile');
};

export class Header extends React.PureComponent {
    constructor(props) {
        super(props);
        const {tld, lang, onlyKUBR} = props;

        this.isTr = lang === 'tr';
        this.isRu = ['ru', 'uk'].includes(lang);
        this.areRestLinksVisible = !onlyKUBR || ['kz', 'ua', 'by', 'ru'].includes(tld);
    }

    componentDidMount() {
        sendRumRaf(RUM_MARKS.FIRST_MEANINGFUL_PAINT_RAF);
    }

    _maybeRenderNavigationLink = (link) => {
        if (!link) {
            return null;
        }
        const {href, to, title, restLinks = []} = link;

        if (href) {
            return (
                <a data-t='dheader-nav:native' className='dheader-navigation-item' href={href} key={href}>
                    {title}
                </a>
            );
        }

        if (to) {
            const {location} = this.props;
            const {pathname = ''} = location || {};
            const Link = location ? RouterLink : FallbackLink;

            return (
                <Link
                    key={to}
                    className={classnames('dheader-navigation-item', {
                        'dheader-navigation-item__active':
                            !restLinks.some((link) => isLinkActive(link, pathname)) &&
                            ((location && typeof location === 'string' && location.includes(to)) ||
                                isLinkActive(link, pathname))
                    })}
                    to={to}
                    data-t='dheader-nav:native'
                >
                    {title}
                </Link>
            );
        }
    };

    _maybeRenderNavigation = () => {
        const {isAvatarPage} = this.props;
        const {
            areRestLinksVisible,
            props: {hideNavigation, tld, isFamilyEnable, isOrderHistoryExp, isSupportWindowEnabled}
        } = this;

        if (hideNavigation || isAvatarPage) {
            return null;
        }

        const links = [
            isSupportWindowEnabled && {
                to: '/support',
                title: i18n('Frontend.link.support')
            },
            isOrderHistoryExp && {
                href: '/order-history',
                title: i18n('Frontend.link.order-history')
            },
            areRestLinksVisible && {
                to: '/profile/services',
                title: i18n('Frontend.link.services')
            },
            areRestLinksVisible &&
                tld === 'ru' && {
                    href: `https://reviews.yandex.${tld}/ugcpub/cabinet`,
                    title: i18n('Frontend.link.ugc')
                },
            isFamilyEnable && {
                to: '/profile/family',
                title: i18n('Frontend.link.family')
            }
        ];

        links.unshift({
            to: '/profile',
            title: i18n('Frontend.link.profile'),
            restLinks: [...links]
        });

        return (
            <div className='dheader-navigation'>
                <div className='dheader-navigation__inner'>{links.map(this._maybeRenderNavigationLink)}</div>
            </div>
        );
    };

    renderUser = () => {
        const {hideUser, authHref} = this.props;

        return hideUser ? (
            <Button view='action' width='max' size='m' as='a' href={authHref}>
                <UserIcon className='dheader-user__icon' />
                {i18n('Frontend.logon')}
            </Button>
        ) : (
            <div className='dheader-user__wrapper'>
                <User />
            </div>
        );
    };

    render() {
        const {props: {location, isLite, isAvatarPage} = {}} = this;
        const Link = location ? RouterLink : FallbackLink;

        if (isLite) {
            return null;
        }

        const headerClassName = classnames('dheader', {
            dheader_avatar: isAvatarPage
        });

        return (
            <div className={headerClassName}>
                <div className='dheader-user'>{this.renderUser()}</div>

                <span className='dheader-logo_container'>
                    <Logo className='yaLogo' />
                    <Link to='/profile' className='dheader-logo_id' />
                </span>

                {this._maybeRenderNavigation()}
            </div>
        );
    }
}

Header.propTypes = {
    onlyKUBR: PropTypes.bool,
    authHref: PropTypes.string,
    tld: PropTypes.string.isRequired,
    lang: PropTypes.string.isRequired,
    location: PropTypes.oneOfType([
        PropTypes.shape({
            search: PropTypes.string.isRequired,
            pathname: PropTypes.string.isRequired,
            href: PropTypes.string
        }),
        PropTypes.string
    ]),
    hideNavigation: PropTypes.bool,
    hideUser: PropTypes.bool,
    isLite: PropTypes.bool,
    isFamilyEnable: PropTypes.bool,
    isOrderHistoryExp: PropTypes.bool,
    isSupportWindowEnabled: PropTypes.bool,
    isPassportNewLogoOnExp: PropTypes.bool,
    isAvatarPage: PropTypes.bool
};

export default connect(mapStateToProps)(Header);

function mapStateToProps(state) {
    const {settings = {}, common = {}, router = {}, header = {}} = state;
    const {tld, language, isLite, passportLinks = {}} = settings;
    const {isFamilyEnable, currentPage} = common;
    const {location} = router;
    const {accounts} = header;
    const {auth: authUrl = ''} = passportLinks;

    return {
        isLite,
        lang: language,
        location,
        onlyKUBR: true,
        tld,
        hideUser: !(accounts && accounts.length),
        authHref: `${authUrl}?retpath=${encodeURIComponent(currentPage)}`,
        isFamilyEnable,
        isOrderHistoryExp: hasExp(state, 'order_history_on'),
        isSupportWindowEnabled: common.isSupportCenterEnable
    };
}
