import React from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';

class Head extends React.Component {
    shouldComponentUpdate() {
        return false;
    }

    render() {
        return (
            <head>
                <title>{this.getTitle()}</title>
                <meta charSet='utf-8' />
                <meta httpEquiv='X-UA-Compatible' content='IE=EmulateIE7,IE=edge' />
                {this.getOpenGraph()}
                {this.getPreload()}
                {this.getMetaIcons()}
                {this.getExtraMetaIcons()}
                {this.getMetaCollections()}
                {this.getCSS()}
                {this.getAuthCustomsCSS()}
                {this.getInlineJS()}
                {this.getJS()}
                {this.getManifest()}
            </head>
        );
    }

    getTitle = () => {
        const {title, settings: {location} = {}} = this.props;

        if (location && location.indexOf('/profile/documents') !== -1) {
            return i18n('Profile2.documents.navigation');
        }

        return title;
    };

    getOpenGraph = () => {
        const {opengraph, settings: {staticPath = '', host = ''} = {}} = this.props;

        return (
            Boolean(opengraph) && (
                <>
                    {Boolean(opengraph.title) && <meta property='og:title' content={opengraph.title} />}
                    {Boolean(opengraph.description) && (
                        <meta property='og:description' content={opengraph.description} />
                    )}
                    {Boolean(opengraph.type) && <meta property='og:type' content={opengraph.type} />}
                    {Boolean(opengraph.image) && (
                        <>
                            <meta property='og:image' content={`${host}/${staticPath}/${opengraph.image.url}`} />
                            <meta property='og:image:type' content={opengraph.image.type} />
                            <meta property='og:image:width' content={opengraph.image.width} />
                            <meta property='og:image:height' content={opengraph.image.height} />
                        </>
                    )}
                </>
            )
        );
    };

    getPreload = () => {
        const {settings: {staticPath} = {}, page, familyPage, withLoadable, language, reactStaticVersion} = this.props;

        return (
            <>
                <link rel='preload' as='style' href={`${staticPath}/css/${page}.cij.css`} />
                {page === 'react.family-lazy' && familyPage && (
                    <link rel='preload' as='style' href={`${staticPath}/css/${familyPage}.${page}.cij.css`} />
                )}
                {!withLoadable && <link rel='preload' as='style' href={`${staticPath}/css/${page}.css`} />}
                {!withLoadable && (
                    <>
                        <link
                            rel='preload'
                            as='script'
                            crossOrigin='anonymous'
                            href={`${staticPath}/js/${page}.vendors.${language}.js`}
                        />
                        <link
                            rel='preload'
                            as='script'
                            crossOrigin='anonymous'
                            href={`${staticPath}/js/${page}.lego.${language}.js`}
                        />
                        <link
                            rel='preload'
                            as='script'
                            crossOrigin='anonymous'
                            href={`${staticPath}/js/${page}.react.${language}.js`}
                        />
                    </>
                )}
                <link crossOrigin='anonymous' href={reactStaticVersion} rel='preload' as='script' />
                <link
                    crossOrigin='anonymous'
                    href='https://yastatic.net/jquery/2.1.4/jquery.min.js'
                    rel='preload'
                    as='script'
                />
            </>
        );
    };

    getCSS = () => {
        const {settings: {staticPath} = {}, familyPage, page, withLoadable} = this.props;

        return (
            <>
                <link rel='stylesheet' type='text/css' href={`${staticPath}/css/${page}.cij.css`} />
                {!withLoadable && <link rel='stylesheet' type='text/css' href={`${staticPath}/css/${page}.css`} />}
                {page === 'react.family-lazy' && familyPage && (
                    <link rel='stylesheet' type='text/css' href={`${staticPath}/css/${familyPage}.${page}.cij.css`} />
                )}
            </>
        );
    };

    getAuthCustomsCSS = () => {
        const {uatraits = {}, settings = {}, loadCustomsStyles, isWebPushModal} = this.props;
        const {authCustomsStatic, authCustomsVersion} = settings;
        const {isTablet, isMobile} = uatraits;

        if (!loadCustomsStyles) {
            return null;
        }

        if (isTablet || !isMobile || isWebPushModal) {
            return (
                <link
                    rel='stylesheet'
                    type='text/css'
                    href={`${authCustomsStatic}/${authCustomsVersion}/passport.auth.customs.css`}
                />
            );
        }
    };

    getInlineJS = () => {
        const {nonce, rumScripts, isYandexInternalNetwork} = this.props;

        return (
            <React.Fragment>
                <script
                    type='text/javascript'
                    nonce={nonce}
                    dangerouslySetInnerHTML={{
                        __html:
                            // eslint-disable-next-line max-len
                            '(function(d, e, c, r, n, w, v, f) {e = d.documentElement; c = "className"; r = "replace"; n = "createElementNS"; f = "firstChild"; w = "http://www.w3.org/2000/svg"; v = d.createElement("div"); v.innerHTML = "<svg/>"; e[c] = e[c][r]("is-js_no", "is-js_yes"); e[c] += " is-inlinesvg_" + ((v[f] && v[f].namespaceURI) === w ? "yes" : "no"); })(document);'
                    }}
                />
                <script
                    type='text/javascript'
                    nonce={nonce}
                    dangerouslySetInnerHTML={{
                        __html:
                            // eslint-disable-next-line max-len
                            'window.onerror=function(d,e,f,g,b){var h=new Image,a=["app=passport"],c="";d&&a.push("msg="+encodeURIComponent(d));e&&a.push("url="+encodeURIComponent(e));(f||g)&&a.push("position="+encodeURIComponent(f+":"+g));if(b){try{c=JSON.stringify(b,Object.getOwnPropertyNames(b))}catch(k){c=b.stack}a.push("error_object="+encodeURIComponent(c))}h.src="/monitoring/onerror?"+a.join("&");return!1};'
                    }}
                />
                {rumScripts && (
                    <script type='text/javascript' nonce={nonce} dangerouslySetInnerHTML={{__html: rumScripts}} />
                )}

                {isYandexInternalNetwork && (
                    <script
                        nonce={nonce}
                        dangerouslySetInnerHTML={{
                            __html: 'window.Passport = {isYandex: true}'
                        }}
                    />
                )}
            </React.Fragment>
        );
    };

    getJS = () => <script crossOrigin='anonymous' src='https://yastatic.net/jquery/2.1.4/jquery.min.js' />;

    getMetaCollections = () => {
        return (
            <React.Fragment>
                <meta name='pdb' content='disable' />
                <meta name='collections' content='disable' />
            </React.Fragment>
        );
    };

    getExtraMetaIcons = () => {
        const {isExtraMetaIconsAvailable, settings = {}} = this.props;
        const {tld, staticPath} = settings;
        const isEn = ['com', 'com.tr'].includes(tld);

        if (!isExtraMetaIconsAvailable) {
            return null;
        }

        return (
            <React.Fragment>
                <meta property='og:image' content={`${staticPath}/i/passp-logos/passport_${isEn ? 'en' : 'ru'}.png`} />
                <meta property='og:image:width' content='129' />
                <meta property='og:image:height' content='29' />
                <meta
                    property='og:image'
                    content={`${staticPath}/i/passp-logos/passport_${isEn ? 'en' : 'ru'}-square.png`}
                />
                <meta property='og:image:width' content='50' />
                <meta property='og:image:height' content='50' />
            </React.Fragment>
        );
    };

    getMetaIcons = () => {
        const {uatraits, display, settings = {}} = this.props;
        const {tld, staticPath} = settings;
        const {isTouch, OSFamily} = uatraits;
        const isEn = ['com', 'com.tr'].includes(tld);
        const imgPath = `${staticPath}/i`;
        const faviconHref = '/favicon.ico';
        const favicon = <link rel='shortcut icon' href={faviconHref} />;

        if (isTouch || display === 'touch') {
            return (
                <React.Fragment>
                    {favicon}
                    <meta name='viewport' content='width=device-width, initial-scale=1.0, user-scalable=no' />
                    <meta name='format-detection' content='telephone=no' />

                    <link
                        rel='apple-touch-icon-precomposed'
                        href={`${imgPath}/apple-touch-icon${isEn ? '-en' : ''}.png`}
                    />
                    <link
                        rel='apple-touch-icon-precomposed'
                        sizes='120x120'
                        href={`${imgPath}/apple-touch-icon-120x120${isEn ? '-en' : ''}.png`}
                    />

                    {OSFamily === 'iOS' ? (
                        <React.Fragment>
                            <link
                                rel='apple-touch-icon-precomposed'
                                sizes='72x72'
                                href={`${imgPath}/apple-touch-icon-72x72${isEn ? '-en' : ''}.png`}
                            />
                            <link
                                rel='apple-touch-icon-precomposed'
                                sizes='76x76'
                                href={`${imgPath}/apple-touch-icon-76x76${isEn ? '-en' : ''}.png`}
                            />
                            <link
                                rel='apple-touch-icon-precomposed'
                                sizes='144x144'
                                href={`${imgPath}/apple-touch-icon-144x144${isEn ? '-en' : ''}.png`}
                            />
                            <link
                                rel='apple-touch-icon-precomposed'
                                sizes='152x152'
                                href={`${imgPath}/apple-touch-icon-152x152${isEn ? '-en' : ''}.png`}
                            />
                        </React.Fragment>
                    ) : (
                        <React.Fragment>
                            <meta name='msapplication-TileColor' content='#da532c' />
                            <meta
                                name='msapplication-TileImage'
                                content={`${imgPath}/apple-touch-icon-144x144${isEn ? '-en' : ''}.png`}
                            />

                            <link
                                rel='icon'
                                sizes='128x128'
                                href={`${imgPath}/touch-icon-128x128${isEn ? '-en' : ''}.png`}
                            />
                            <link
                                rel='icon'
                                sizes='192x192'
                                href={`${imgPath}/touch-icon-192x192${isEn ? '-en' : ''}.png`}
                            />
                            <link
                                rel='apple-touch-icon'
                                sizes='128x128'
                                href={`${imgPath}/touch-icon-128x128${isEn ? '-en' : ''}.png`}
                            />
                            <link
                                rel='apple-touch-icon-precomposed'
                                sizes='128x128'
                                href={`${imgPath}/touch-icon-128x128${isEn ? '-en' : ''}.png`}
                            />
                            <link
                                rel='apple-touch-icon-precomposed'
                                href={`${imgPath}/apple-touch-icon-precomposed${isEn ? '-en' : ''}.png`}
                            />
                            <link
                                rel='apple-touch-icon-precomposed'
                                sizes='120x120'
                                href={`${imgPath}/apple-touch-icon-120x120-precomposed${isEn ? '-en' : ''}.png`}
                            />
                        </React.Fragment>
                    )}
                </React.Fragment>
            );
        }

        return favicon;
    };

    getManifest = () => {
        const {settings: {location} = {}} = this.props;

        if (location && location.indexOf('/profile/documents/pp') !== -1) {
            return <link rel='manifest' href={`/documents-manifest.json`} />;
        }
    };
}

Head.getOpenHtmlTag = (props) => {
    const {page, language, uatraits = {}, display, settings = {}} = props;
    const {isTouch, isTablet} = uatraits;
    const headClassName = classnames('is-js_no', {
        'is-touch': (isTouch && !isTablet) || display === 'touch',
        'is-tablet': isTablet
    });

    return `<html lang="${language}" dir="${settings.dir || 'ltr'}" data-page-type="${page}" class="${headClassName}">`;
};

Head.propTypes = {
    uatraits: PropTypes.shape({
        isTablet: PropTypes.bool,
        isTouch: PropTypes.bool,
        OSFamily: PropTypes.string
    }),
    settings: PropTypes.shape({
        staticPath: PropTypes.string.isRequired,
        tld: PropTypes.string.isRequired
    }),
    opengraph: PropTypes.object,
    familyPage: PropTypes.string,
    title: PropTypes.string.isRequired,
    page: PropTypes.string.isRequired,
    language: PropTypes.string.isRequired,
    reactStaticVersion: PropTypes.string.isRequired,
    display: PropTypes.string,
    nonce: PropTypes.string.isRequired,
    rumScripts: PropTypes.string,
    isYandexInternalNetwork: PropTypes.bool,
    isWebPushModal: PropTypes.bool,
    isExtraMetaIconsAvailable: PropTypes.bool,
    loadCustomsStyles: PropTypes.bool,
    withLoadable: PropTypes.bool
};

export {Head};
