import React from 'react';
import {Provider} from 'react-redux';
import ReactStaticVersion from '@yandex-int/react-static-version';
import {renderToStaticMarkup, renderToNodeStream} from 'react-dom/server';
import {ConnectedRouter} from 'connected-react-router';
import {ChunkExtractor} from '@loadable/server';
import {getDefaultLayoutProps} from '@blocks/selectors';
import {getRumResTimingScipt} from '@plibs/rum-counter/utils';
import {Head} from '@components/Layout/Head';
import {Body} from '@components/Layout/Body';

export const renderServer = (
    res,
    {
        App,
        configureStore,
        title,
        withProvider = true,
        withRouter = false,
        withLoadable = false,
        history,
        loadableStatsFile,
        extraProps = {}
    }
) => {
    const defaultLayoutProps = getDefaultLayoutProps(res.locals);
    const {page, nonce} = defaultLayoutProps;
    const {development = {}, production = {}} = ReactStaticVersion;
    const layoutProps = {
        ...defaultLayoutProps,
        withLoadable,
        title,
        ...extraProps,
        reactStaticVersion:
            process.env.NODE_ENV === 'development'
                ? development.reactWithDomAndPolyfills
                : production.reactWithDomAndPolyfills
    };

    const head = renderToStaticMarkup(<Head {...layoutProps} />);
    const html = Head.getOpenHtmlTag(layoutProps);
    const body = Body.getOpenBodyTag(layoutProps);
    const store = withProvider ? configureStore(res.locals.store, history) : res.locals.store;

    const extractor = withLoadable ? new ChunkExtractor({statsFile: loadableStatsFile, entrypoints: [page]}) : null;

    res.type('html');
    res.write(`<!doctype html>${html}${head}${body}`, 'utf-8');

    const Component = withProvider ? (
        <Provider store={store}>
            {withRouter ? (
                <ConnectedRouter history={history}>
                    <App />
                </ConnectedRouter>
            ) : (
                <App />
            )}
        </Provider>
    ) : (
        <App {...store} />
    );
    const ComponentWithBody = <Body {...layoutProps}>{Component}</Body>;
    const stream = renderToNodeStream(withLoadable ? extractor.collectChunks(ComponentWithBody) : ComponentWithBody);

    stream.on('end', () => {
        if (withLoadable) {
            const scriptTags = extractor.getScriptTags({crossorigin: 'anonymous'});
            const linkTags = extractor.getLinkTags({crossorigin: 'anonymous'});
            const styleTags = extractor.getStyleTags({crossorigin: 'anonymous'});

            return res.end(
                `${scriptTags}${linkTags}${styleTags}${getRumResTimingScipt(
                    nonce,
                    extractor.getScriptElements(),
                    extractor.getStyleElements()
                )}</body></html>`
            );
        }

        return res.end('</body></html>');
    });
    stream.pipe(res);
};
