import React, {useEffect, useCallback} from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import {AM_SHOW_DEBUG_INFO_CLICKS_COUNT} from '@blocks/authv2/nativeMobileApi';

import './Logo.styl';

const b = cn('Logo');

export const Logo = ({
    onShowDebugInfo,
    lang,
    tld,
    isAm,
    isStatic,
    whitelabelLogo,
    fullscreenLogo,
    authCustomsStatic
}) => {
    const logoClicksCount = React.useRef(0);

    useEffect(() => {
        if (isAm) {
            const windowClickListener = () => {
                logoClicksCount.current = 0;
            };

            window.addEventListener('click', windowClickListener);

            return () => {
                window.removeEventListener('click', windowClickListener);
            };
        }
    }, [isAm, logoClicksCount, onShowDebugInfo]);

    const onLogoClick = useCallback(
        (event) => {
            event.stopPropagation();
            event.preventDefault();
            if (isAm) {
                logoClicksCount.current++;
                if (logoClicksCount.current >= AM_SHOW_DEBUG_INFO_CLICKS_COUNT) {
                    onShowDebugInfo();
                }
            }
        },
        [isAm, logoClicksCount, onShowDebugInfo]
    );

    const renderCustomLogo = ({asset, width, height} = {}, asImage = false) => {
        const commonProps = {onClick: onLogoClick, className: b({asImage})};
        const assetUrl = `${authCustomsStatic}/assets/${asset}`;

        return asImage ? (
            <img {...commonProps} src={assetUrl} />
        ) : (
            <span {...commonProps} style={{backgroundImage: `url('${assetUrl}')`, width, height}} />
        );
    };

    if (whitelabelLogo) {
        return renderCustomLogo(whitelabelLogo);
    }

    if (fullscreenLogo) {
        return renderCustomLogo(fullscreenLogo, true);
    }

    const isRu = !lang || ['ru', 'uk'].includes(lang);
    const Component = isStatic ? 'span' : 'a';
    const options = {};

    if (!isStatic) {
        options.href = `//yandex.${tld}`;
        options.tabIndex = '-1';
    } else {
        options.onClick = onLogoClick;
    }

    return <Component {...options} className={b({en: !isRu, ya: true})} />;
};

Logo.propTypes = {
    className: PropTypes.string,
    lang: PropTypes.string,
    size: PropTypes.oneOf(['s', 'm', 'l']),
    tld: PropTypes.string,
    isWhite: PropTypes.bool,
    isStatic: PropTypes.bool,
    isAm: PropTypes.bool,
    whitelabelLogo: PropTypes.shape({
        asset: PropTypes.string,
        width: PropTypes.number,
        height: PropTypes.number
    }),
    fullscreenLogo: PropTypes.shape({
        asset: PropTypes.string,
        width: PropTypes.number,
        height: PropTypes.number
    }),
    authCustomsStatic: PropTypes.string,
    logoType: PropTypes.string,
    onShowDebugInfo: PropTypes.func.isRequired
};
