import React, {useEffect, useMemo} from 'react';
import PropTypes from 'prop-types';
import {useSelector, shallowEqual} from 'react-redux';
import {ShieldOutline} from '@yandex/ui-icons';
import {sendRumRaf, RUM_MARKS} from '@plibs/rum-counter/utils';
import {hasExp} from '@blocks/selectors';
import {Page} from '@yandex-id/components';
import {Link} from 'react-router-dom';
import {PageHeaderPlus} from './Plus';
import {User} from '@components/User';
import {getBonusUrl, getPasswordUrl, ROUTES_WITH_RELOAD} from './Wrapper.util';

import {
    PageWrapperIconUGC,
    PageWrapperIconHelp,
    PageWrapperIconFamily,
    PageWrapperIconSupport,
    PageWrapperIconAccount,
    PageWrapperIconPayments,
    PageWrapperIconServices,
    PageWrapperIconBonusCard,
    PageWrapperIconPasswordsCard,
    PageWrapperIconDocuments,
    PageWrapperIconPersonal,
    PageWrapperIconMain,
    PageWrapperIconFamilyNew
} from './Icon';

const ICONS = {
    ugc: <PageWrapperIconUGC />,
    help: <PageWrapperIconHelp />,
    family: <PageWrapperIconFamily />,
    support: <PageWrapperIconSupport />,
    account: <PageWrapperIconAccount />,
    payments: <PageWrapperIconPayments />,
    services: <PageWrapperIconServices />,
    bonusCard: <PageWrapperIconBonusCard />,
    passwordsCard: <PageWrapperIconPasswordsCard />,
    documents: <PageWrapperIconDocuments />,
    security: <ShieldOutline />,
    personal: <PageWrapperIconPersonal />,
    main: <PageWrapperIconMain />,
    familyNew: <PageWrapperIconFamilyNew />
};

export const PageWrapper = ({unwrap, children, isProfile, isNavigationHidden, isHeaderHidden}) => {
    const {
        tld,
        lang,
        helpLink,
        pathname,
        isNewLayout,
        isHeaderHidden: isPageHeaderHidden,
        isFamilyEnabled,
        isUserHistoryEnabled,
        isSupportWindowEnabled,
        bonusUrl,
        passwordUrl,
        routerLinkKey,
        isDocumentsEnabled,
        isFinanceEnabled,
        isSecurityEnabled,
        isHelpdeskEnabled,
        isPersonalEnabled,
        isNewNavigationEnabled
    } = useSelector((state) => {
        const {settings = {}, common = {}, router = {}} = state;
        const newPageEnabled = settings.ua && settings.ua.BrowserName !== 'MSIE';

        // passport-<page>-exp только для Яндексоидов, passpost-all-* – для всех
        const isFinanceEnabled =
            (hasExp(state, 'passport-finance-exp') && common.isYandexoid) || hasExp(state, 'passport-all-finance-exp');
        const isSecurityEnabled =
            (hasExp(state, 'passport-security-exp') && common.isYandexoid) ||
            hasExp(state, 'passport-all-security-exp');
        const isHelpdeskEnabled =
            (hasExp(state, 'passport-helpdesk-exp') && common.isYandexoid) ||
            hasExp(state, 'passport-all-helpdesk-exp');
        const isPersonalEnabled =
            (hasExp(state, 'passport-personal-exp') && common.isYandexoid) ||
            hasExp(state, 'passport-all-personal-exp');
        const lang = settings.language || settings.lang;

        const isNewNavigationEnabled =
            (hasExp(state, 'passport-new-navigation-for-staff-exp') && common.isYandexoid) ||
            (lang === 'ru' &&
                newPageEnabled &&
                hasExp(state, 'passport-new-navigation-exp') &&
                hasExp(state, 'passport-new-lk-exp'));

        const stateData = {
            tld: settings.tld,
            isNewLayout: settings.isNewLayout,
            isFamilyEnabled: common.isFamilyEnable,
            isHeaderHidden: isHeaderHidden || settings.isLite || false,
            lang,
            pathname: (router.location || {}).pathname || '',
            isUserHistoryEnabled: hasExp(state, 'order_history_on'),
            isSupportWindowEnabled: common.isSupportCenterEnable,
            isDocumentsEnabled: hasExp(state, 'passport-documents-exp'),
            helpLink: ((settings.help || {}).passport || '#').replace('%tld%', settings.tld),
            isFinanceEnabled: newPageEnabled && isFinanceEnabled,
            isSecurityEnabled: newPageEnabled && isSecurityEnabled,
            isHelpdeskEnabled: newPageEnabled && isHelpdeskEnabled,
            isPersonalEnabled: newPageEnabled && isPersonalEnabled,
            isNewNavigationEnabled: newPageEnabled && isNewNavigationEnabled
        };

        const routerLinkKey = ROUTES_WITH_RELOAD.indexOf(stateData.pathname) > -1 ? 'href' : isProfile ? 'to' : 'href';

        return {
            ...stateData,
            routerLinkKey,
            bonusUrl: hasExp(state, 'passport-bonuscards-exp') ? getBonusUrl(settings.ua, routerLinkKey) : undefined,
            passwordUrl: getPasswordUrl(settings.ua, routerLinkKey)
        };
    }, shallowEqual);

    const navigationLinks = useMemo(() => {
        if (isNavigationHidden) {
            return [];
        }

        if (isNewNavigationEnabled) {
            return [
                {
                    href: '/',
                    title: i18n('Frontend.link.main'),
                    icon: ICONS.main
                },
                {
                    href: '/personal',
                    title: i18n('Frontend.link.personal'),
                    icon: ICONS.personal
                },
                {
                    href: '/pay',
                    title: i18n('Frontend.link.pay'),
                    icon: ICONS.payments
                },
                {
                    [routerLinkKey]: '/profile/family',
                    title: i18n('Frontend.link.family.short'),
                    icon: ICONS.familyNew
                },
                {
                    href: '/security',
                    title: i18n('Frontend.link.security'),
                    icon: ICONS.security
                }
            ];
        }

        const restLinksVisible = ['kz', 'ua', 'by', 'ru'].includes(tld);

        return [
            {[routerLinkKey]: '/profile', title: i18n('Frontend.link.account'), icon: ICONS.account},
            isPersonalEnabled && {
                href: '/personal',
                title: i18n('Frontend.link.personal'),
                icon: ICONS.personal
            },
            isUserHistoryEnabled && {
                href: isFinanceEnabled ? '/finance' : '/order-history',
                title: i18n('Frontend.link.order-history'),
                icon: ICONS.payments
            },
            isFamilyEnabled && {
                [routerLinkKey]: '/profile/family',
                title: i18n('Frontend.link.family.short'),
                icon: ICONS.family
            },
            isSecurityEnabled && {
                href: '/security',
                title: i18n('Frontend.link.security'),
                icon: ICONS.security
            },
            isSupportWindowEnabled && {
                href: isHelpdeskEnabled ? '/helpdesk' : '/support',
                title: i18n('Frontend.link.support'),
                icon: ICONS.support
            },
            restLinksVisible && {
                [routerLinkKey]: '/profile/services',
                title: i18n('Frontend.link.services'),
                icon: ICONS.services
            },
            restLinksVisible &&
                tld === 'ru' && {
                    href: `https://reviews.yandex.${tld}/ugcpub/cabinet`,
                    title: i18n('Frontend.link.ugc'),
                    icon: ICONS.ugc
                }
        ].filter(Boolean);
    }, [
        tld,
        isNavigationHidden,
        isFamilyEnabled,
        isUserHistoryEnabled,
        isSupportWindowEnabled,
        routerLinkKey,
        isFinanceEnabled,
        isHelpdeskEnabled,
        isSecurityEnabled,
        isPersonalEnabled,
        isNewNavigationEnabled
    ]);

    const specialNavigationLinks = useMemo(() => {
        if (isNavigationHidden || isNewNavigationEnabled) {
            return [];
        }

        const links = [];

        if (isDocumentsEnabled) {
            links.push({
                [routerLinkKey]: '/profile/documents',
                title: i18n('Profile2.documents.navigation'),
                icon: ICONS.documents
            });
        }

        if (passwordUrl) {
            links.push({
                ...passwordUrl,
                title: i18n('Profile2.passwords.tabName'),
                icon: ICONS.passwordsCard
            });
        }

        if (bonusUrl) {
            links.push({
                ...bonusUrl,
                title: i18n('Profile2.bonus.tabName'),
                icon: ICONS.bonusCard
            });
        }

        return links.filter(Boolean);
    }, [bonusUrl, passwordUrl, isNavigationHidden, routerLinkKey, isDocumentsEnabled, isNewNavigationEnabled]);

    const otherNavigationLinks = useMemo(() => {
        if (isNewNavigationEnabled) {
            return [
                {
                    href: '/helpdesk',
                    title: i18n('Frontend.link.support.new'),
                    icon: ICONS.support
                },
                {
                    href: helpLink,
                    title: i18n('Frontend.footer.reference'),
                    icon: ICONS.help
                }
            ];
        }

        return [
            {
                href: helpLink,
                title: i18n('Frontend.footer.reference'),
                icon: ICONS.help,
                blank: true
            }
        ].filter(Boolean);
    }, [helpLink, isNewNavigationEnabled]);

    useEffect(() => {
        sendRumRaf(RUM_MARKS.FIRST_MEANINGFUL_PAINT_RAF);
    }, []);

    return isNewLayout && !unwrap ? (
        <Page
            tld={tld}
            lang={lang}
            pathname={pathname}
            isProfile={isProfile}
            isHeaderHidden={isPageHeaderHidden}
            navigationLinks={navigationLinks}
            otherNavigationLinks={otherNavigationLinks}
            specialNavigationLinks={specialNavigationLinks}
            linkComponent={Link}
            plus={<PageHeaderPlus />}
            user={<User />}
        >
            {children}
        </Page>
    ) : (
        children
    );
};

PageWrapper.propTypes = {
    unwrap: PropTypes.bool,
    children: PropTypes.node,
    isProfile: PropTypes.bool,
    isNavigationHidden: PropTypes.bool,
    isHeaderHidden: PropTypes.bool
};
