import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {Modal} from '@components/Modal';
import {Input} from '@components/Input';
import {Button} from '@components/Button';
import {Captcha} from '@components/Captcha';
import {checkHuman, validateForm, hideDomik} from './actions';

import {cn} from '@bem-react/classname';

import './RequestDataDomik.styl';

const b = cn('RequestDataDomik');

const errorMessages = {
    password: {
        undefined: i18n('_AUTH_.errors.required'),
        empty: i18n('_AUTH_.errors.required'),
        notMatched: i18n('Profile.domik.password-not-matched')
    },
    captcha: {
        undefined: i18n('_AUTH_.errors.required'),
        empty: i18n('_AUTH_.errors.required'),
        notMatched: i18n('Profile.domik.captcha-not-matched')
    }
};

class Domik extends Component {
    state = {
        password: '',
        captcha: '',
        errorIsShowed: false
    };

    submitForm = (e) => {
        e.preventDefault();
        const data = this.getFieldValues();

        this.props.dispatch(validateForm(data));
        this.props.dispatch(checkHuman(data));
        this.setState({errorIsShowed: true});
    };

    getFieldValues = () => {
        const domik = this.props.domik;
        const data = {};

        if (domik.requestCaptcha) {
            data.captcha = this.state.captcha;
        }

        if (domik.requestPassword) {
            data.password = this.state.password;
        }

        return data;
    };

    handleInput = (event) => {
        const {target = {}} = event;
        const {value, name} = target;

        this.setState({
            [name]: value,
            errorIsShowed: false
        });
    };

    hideDomik = () => {
        this.props.dispatch(hideDomik());
    };

    onHistoryChange = () => {
        this.hideDomik();
    };

    componentDidMount() {
        const {requestPassword, requestCaptcha} = this.props.domik;

        if (requestPassword) {
            this.password.focus();
        } else if (requestCaptcha) {
            this.captcha.focus();
        }

        window.addEventListener('popstate', this.onHistoryChange);
    }

    componentWillUnmount() {
        window.removeEventListener('popstate', this.onHistoryChange);
    }

    render() {
        const props = this.props;
        const {isMobileCaptcha, captcha, lang, domik, is2faEnabled, isMobile} = props;
        const {requestPassword, requestCaptcha, errors} = domik;
        const logoLang = ['ru', 'uk'].indexOf(lang) > -1 ? 'ru' : 'en'; // uk это украинский!!!
        const {errorIsShowed} = this.state;
        const formText = is2faEnabled
            ? i18n('Frontend.otp_input_description')
            : i18n('Frontend.auth.verify.password.subheader');
        const inputPlaceholderText = is2faEnabled ? i18n('Frontend.onetime_password') : i18n('Profile.domik.password');

        return (
            <div className={b()}>
                <div className={b('logo', {logoLang})} />
                <form onSubmit={this.submitForm}>
                    <div className={b('input', {'g-hide': !requestPassword})}>
                        <label className={b('label', {password: true})} htmlFor='password'>
                            {formText}
                        </label>
                        <Input
                            id='password'
                            name='password'
                            type='password'
                            size='m'
                            controlRef={(password) => (this.password = password)}
                            value={this.state.password}
                            placeholder={inputPlaceholderText}
                            onChange={this.handleInput}
                        />
                        {errorIsShowed && errors.password && (
                            <div className={b('errorBlock')}>{errorMessages.password[errors.password]}</div>
                        )}
                    </div>
                    {requestCaptcha && (
                        <div>
                            <div className={b('captcha')}>
                                <Captcha
                                    isMobile={isMobileCaptcha}
                                    lang={lang}
                                    introSound={captcha.introSound}
                                    captchaSound={captcha.captchaSound}
                                    imageUrl={captcha.imageUrl}
                                    loading={captcha.loading}
                                    loadingAudio={captcha.loadingAudio}
                                    playing={captcha.playing}
                                    type={captcha.type}
                                    dispatch={props.dispatch}
                                    visible={requestCaptcha}
                                />
                            </div>
                            <div className={b('input')}>
                                <label className={b('label')}>{i18n('Profile.domik.enter-symbols')}</label>
                                <Input
                                    id='captcha'
                                    name='captcha'
                                    size='m'
                                    controlRef={(captcha) => (this.captcha = captcha)}
                                    value={this.state.captcha}
                                    onChange={this.handleInput}
                                />
                                {errorIsShowed && errors.captcha && (
                                    <div className='p-control-error-block'>{errorMessages.captcha[errors.captcha]}</div>
                                )}
                            </div>
                        </div>
                    )}
                    <div className={b('submitWrapper')}>
                        <Button width='max' view='action' size='m' type='submit'>
                            {i18n('Profile.domik.confirm')}
                        </Button>
                    </div>
                    <div className={b('btnsWrapper')}>
                        {requestPassword && (
                            <Button view='pseudo' size='m' url='/passport?mode=restore' type='link'>
                                {i18n('Profile.domik.remember-password')}
                            </Button>
                        )}
                        <div
                            className={b('cancelButton', {
                                wideBtn: !requestPassword || isMobile,
                                narrowBtn: requestPassword && !isMobile
                            })}
                        >
                            <Button onClick={this.hideDomik} view='pseudo' size='m'>
                                {i18n('_AUTH_.common.cancel')}
                            </Button>
                        </div>
                    </div>
                </form>
            </div>
        );
    }
}

Domik.propTypes = {
    dispatch: PropTypes.func.isRequired,
    domik: PropTypes.shape({
        requestPassword: PropTypes.bool.isRequired,
        requestCaptcha: PropTypes.bool.isRequired,
        errors: PropTypes.object
    }).isRequired,
    captcha: PropTypes.object.isRequired,
    lang: PropTypes.string.isRequired,
    isMobileCaptcha: PropTypes.bool.isRequired,
    is2faEnabled: PropTypes.bool,
    isMobile: PropTypes.bool.isRequired
};

export class RequestDataDomik extends Component {
    render() {
        const props = this.props;
        const {isMobile} = this.props;
        const {requestPassword, requestCaptcha} = props.domik;

        if (isMobile && (requestPassword || requestCaptcha)) {
            return (
                <div className={b('mobile')}>
                    <Domik {...props} />
                </div>
            );
        }

        return (
            <Modal className='domik_password' visible={requestPassword || requestCaptcha} zIndexGroupLevel={90}>
                {(requestPassword || requestCaptcha) && <Domik {...props} />}
            </Modal>
        );
    }
}

RequestDataDomik.propTypes = {
    domik: PropTypes.shape({
        requestPassword: PropTypes.bool.isRequired,
        requestCaptcha: PropTypes.bool.isRequired
    }).isRequired,
    lang: PropTypes.string,
    isMobileCaptcha: PropTypes.bool,
    isMobile: PropTypes.bool,
    is2faEnabled: PropTypes.bool
};
