import React, {Fragment, PureComponent} from 'react';
import PropTypes from 'prop-types';
import {Button} from '@components/Button';
import {FIELDS_NAMES, Field} from '@components/Field';

import {cn} from '@bem-react/classname';
import './ConfigureEmail.styl';

const b = cn('ConfigureEmail');

const Email = ({email, isAbleToRemove, isEditable, handleDelete, handleRestore, hasMarkedDeleted}) => {
    const showDelete = isEditable && isAbleToRemove && !hasMarkedDeleted;

    return (
        <>
            <span className={b('emailAddress')}>
                {email}
                {hasMarkedDeleted && (
                    <em className={b('markedDeleted')}>{i18n('ProfileSecurity.security.email_deleted_mark')}</em>
                )}
            </span>
            {showDelete && (
                <button
                    data-t='delete-email-btn'
                    type='button'
                    className={b('deleteBtn')}
                    value={email}
                    onClick={handleDelete}
                    aria-label={i18n('ProfileSecurity.security.email_delete-btn')}
                />
            )}
            {hasMarkedDeleted && (
                <button
                    data-t='cancel-delete-email-btn'
                    type='button'
                    onClick={handleRestore}
                    className={b('cancelDeleteBtn')}
                    value={email}
                >
                    {i18n('_AUTH_.common.cancel')}
                </button>
            )}
        </>
    );
};

Email.propTypes = {
    email: PropTypes.string.isRequired,
    isEditable: PropTypes.bool,
    hasMarkedDeleted: PropTypes.bool,
    handleDelete: PropTypes.func,
    handleRestore: PropTypes.func,
    isAbleToRemove: PropTypes.bool
};

export class ConfigureEmail extends PureComponent {
    sendCode = (event) => {
        if (event) {
            event.preventDefault();
        }

        this.props.sendConfirmationCode();
    };

    checkCode = (event) => {
        if (event) {
            event.preventDefault();
        }

        this.props.checkConfirmationCode();
    };

    deleteEmail = (event) => {
        const emailToDelete = event && event.currentTarget && event.currentTarget.value;

        this.props.markEmailDeleted(emailToDelete);
    };

    cancelDelete = (event) => {
        const emailToRestore = event && event.currentTarget && event.currentTarget.value;

        this.props.unmarkEmailDeleted(emailToRestore);
    };

    renderListOfEmails = () => {
        const {emailsList, emailsToDelete, isLiteUser, hasPassword, login} = this.props;
        const isAbleToRemove = (email) => (!isLiteUser && hasPassword) || (isLiteUser && login !== email);

        return (
            <Fragment>
                <ul className={b('emailList')}>
                    {emailsList.map((email, index) => {
                        const hasMarkedDeleted = emailsToDelete.includes(email);

                        return (
                            <li key={index} className={b({hasMarkedDeleted})}>
                                <Email
                                    email={email}
                                    isAbleToRemove={isAbleToRemove(email)}
                                    isEditable={true}
                                    handleDelete={this.deleteEmail}
                                    handleRestore={this.cancelDelete}
                                    hasMarkedDeleted={hasMarkedDeleted}
                                />
                            </li>
                        );
                    })}
                </ul>
            </Fragment>
        );
    };

    renderCodeConfirmation = () => {
        const {codeFieldError, isLoading} = this.props;

        return (
            <form action='#' method='post' className={b('codeForm')} onSubmit={this.checkCode}>
                <div className={b('enterCodeInput')}>
                    <Field
                        size='l'
                        error={codeFieldError}
                        placeholder={i18n('ProfileDelete.code-field_email-placeholder')}
                        name={FIELDS_NAMES.EMAIL_CODE}
                        options={{inputmode: 'numeric'}}
                    />
                </div>
                <button
                    data-t='send-code-again-btn'
                    className={b('sendCodeAgainBtn')}
                    type='button'
                    onClick={this.sendCode}
                    disabled={isLoading}
                >
                    {i18n('ProfileSecurity.security.ask_code-again_btn')}
                </button>
            </form>
        );
    };

    render() {
        const {emailsList, hasCodeSent, emailFieldError, isLoading} = this.props;
        const hasEmails = Boolean(emailsList.length);

        if (hasCodeSent) {
            return this.renderCodeConfirmation();
        }

        return (
            <div className={b('container')}>
                <form action='#' method='post' className={b('form')} onSubmit={this.sendCode} noValidate={true}>
                    <div className={b('enterEmailInput')}>
                        <Field
                            size='l'
                            error={emailFieldError}
                            placeholder='name@example.com'
                            name={FIELDS_NAMES.EMAIL}
                            type='email'
                        />
                    </div>
                    <Button
                        dataT='send-code-btn'
                        size='l'
                        view='action'
                        type='button'
                        onClick={this.sendCode}
                        progress={isLoading}
                    >
                        {i18n('ProfileSecurity.security.add_email-btn')}
                    </Button>
                </form>

                {hasEmails && (
                    <div className={b(emailsList)}>
                        <h3 className={b('listTitle')}>{i18n('ProfileSecurity.security.added_emails')}</h3>
                        {this.renderListOfEmails()}
                    </div>
                )}
            </div>
        );
    }
}

ConfigureEmail.propTypes = {
    isMobile: PropTypes.bool,
    emailsList: PropTypes.array.isRequired,
    hasCodeSent: PropTypes.bool.isRequired,
    sendConfirmationCode: PropTypes.func.isRequired,
    checkConfirmationCode: PropTypes.func.isRequired,
    markEmailDeleted: PropTypes.func.isRequired,
    unmarkEmailDeleted: PropTypes.func.isRequired,
    emailsToDelete: PropTypes.array,
    codeFieldError: PropTypes.object,
    emailFieldError: PropTypes.object,
    isLoading: PropTypes.bool,
    isLiteUser: PropTypes.bool,
    isAbleToRemove: PropTypes.bool,
    hasPassword: PropTypes.bool,
    login: PropTypes.string
};
