import React, {useRef, useState, useEffect, useCallback} from 'react';
import {useSelector, shallowEqual} from 'react-redux';
import {cn} from '@bem-react/classname';

import {Qr} from './Qr';
import {Intro} from './Intro';
import {Button} from './Button';

import './DeepSync.styl';

const b = cn('DeepSync');

const BRO_LINK = 'https://browser.yandex.ru/mobile?from=deep_sync';
const YA_LINK = 'https://yandex.ru/promo/searchapp/desktop?from=deep_sync';
const SCAN_DESCRIPTION = i18n('DeepSync.scan.description')
    .replace('%camera%', `<span class="${b('camera')}"></span>`)
    .replace('%ya_href%', YA_LINK)
    .replace('%bro_href%', BRO_LINK);
const INTRO_DESCRIPTION = i18n('DeepSync.intro.description')
    .replace('%break%', '<br>')
    .replace('%break%', '<br>');

export const DeepSync = () => {
    const timer = useRef();
    const {authUrl, isMobile, error, magicTrack, authChallengeUrl} = useSelector(
        ({common: {authUrl, authChallengeUrl}, deepSync: {error, magicTrack}, settings: {isMobile}}) => ({
            error,
            isMobile,
            authUrl,
            magicTrack,
            authChallengeUrl
        }),
        shallowEqual
    );
    const [visibleCount, setVisibleCount] = useState(0);
    const onQrError = useCallback(
        (error, isState) => {
            location.href = `${error === 'auth_challenge' ? authChallengeUrl : authUrl}${
                isState ? `&track_id=${magicTrack}` : ''
            }`;
        },
        [authUrl, magicTrack, authChallengeUrl]
    );
    const onQrTimeout = useCallback(() => {
        location.href = authUrl;
    }, [authUrl]);
    const blocksLength = isMobile ? 1 : 2;
    const hasError = Boolean(error);

    useEffect(() => {
        if (error) {
            location.href = authUrl;
        }
    }, [error, authUrl]);

    useEffect(() => {
        if (visibleCount > 0) {
            const scrollHeight = document.body.scrollHeight || document.scrollingElement.scrollHeight;

            window.scrollTo({
                left: 0,
                top: visibleCount > 1 ? (scrollHeight - window.innerHeight) * 0.5 : 9999,
                behavior: 'smooth'
            });
        }

        if (visibleCount < blocksLength) {
            timer.current = setTimeout(() => setVisibleCount((c) => c + 1), !visibleCount ? 0 : 1500);

            return () => clearTimeout(timer.current);
        }
    }, [blocksLength, visibleCount]);

    return (
        <div className={b()}>
            <div
                className={b(
                    'toggle',
                    {
                        active: visibleCount > 0
                    },
                    [b('mb_12')]
                )}
            >
                <div className={b('head')}>{i18n('DeepSync.fav')}</div>
                <div
                    className={b(
                        'block',
                        {
                            top: true,
                            bbr: true
                        },
                        [b('mb_4')]
                    )}
                >
                    <div className={b('intro')}>
                        <Intro active={visibleCount > 0} />
                    </div>
                    <div dangerouslySetInnerHTML={{__html: INTRO_DESCRIPTION}} />
                </div>
                <Button className={b('button')} dependent={true} href={authUrl} blank={!isMobile}>
                    {i18n('DeepSync.enter')}
                </Button>
            </div>
            {!isMobile && (
                <>
                    <div
                        className={b('toggle', {
                            active: visibleCount > 1
                        })}
                    >
                        <div
                            className={b(
                                'block',
                                {
                                    top: true,
                                    bbr: hasError
                                },
                                [hasError && b('mb_4')]
                            )}
                            dangerouslySetInnerHTML={{__html: SCAN_DESCRIPTION}}
                        />
                        {!hasError && (
                            <div
                                className={b(
                                    'block',
                                    {
                                        qr: true
                                    },
                                    [b('mb_10')]
                                )}
                            >
                                <div className={b('qr')}>
                                    <Qr onError={onQrError} onTimeout={onQrTimeout} active={visibleCount > 1} />
                                </div>
                            </div>
                        )}
                        <Button className={b('button')} href={authUrl} blank={!isMobile}>
                            {i18n('DeepSync.or-by-phone')}
                        </Button>
                    </div>
                </>
            )}
        </div>
    );
};
