import {useRef, useEffect, useCallback} from 'react';
import api from '@blocks/api';
import {useSelector, useDispatch, shallowEqual} from 'react-redux';
import {setLoading, updateMagicTokens} from '../actions';

export const useLogic = ({active, onError, onTimeout}) => {
    const dispatch = useDispatch();
    const updateDelayRef = useRef(300);
    const reAuthRetriesRef = useRef(0);
    const timeoutErrorCountRef = useRef(0);
    const reAuthTimeoutRef = useRef(null);
    const updateTimeoutRef = useRef(null);
    const timeoutTimeoutRef = useRef(null);
    const intervalTimeoutRef = useRef(null);
    const {csrf, isLoading, magicCSRF, magicTrack} = useSelector(
        ({deepSync: {magicTrack, magicCSRF, isLoading}, common: {csrf, authUrl}}) => ({
            csrf,
            authUrl,
            isLoading,
            magicCSRF,
            magicTrack
        }),
        shallowEqual
    );

    const clearIntervals = useCallback(() => {
        clearTimeout(reAuthTimeoutRef.current);
        clearTimeout(updateTimeoutRef.current);
        clearTimeout(timeoutTimeoutRef.current);
        clearTimeout(intervalTimeoutRef.current);
    }, []);

    const sendError = useCallback(
        (...args) => {
            clearIntervals();
            onError(...args);
        },
        [onError, clearIntervals]
    );

    const reset = useCallback(() => {
        timeoutErrorCountRef.current = 0;
        reAuthRetriesRef.current = 0;
        updateDelayRef.current = 300;
        intervalTimeoutRef.current = setTimeout(() => {
            updateDelayRef.current = 1000;
        }, 6e4);
        timeoutTimeoutRef.current = setTimeout(() => {
            clearIntervals();
            onTimeout();
        }, 6e5);
    }, [onTimeout, clearIntervals]);

    const reAuthPasswordSubmit = useCallback(() => {
        dispatch(setLoading(true));
        return api
            .request('/registration-validations/reAuthPasswordSubmit', {csrf_token: csrf})
            .done(({csrf_token: newMagicCSRF, track_id: newMagicTrack}) =>
                dispatch(updateMagicTokens({magicCSRF: newMagicCSRF, magicTrack: newMagicTrack}))
            )
            .catch(() => {
                if (reAuthRetriesRef.current > 2) {
                    sendError('global');
                } else {
                    ++reAuthRetriesRef.current;
                    reAuthTimeoutRef.current = setTimeout(reAuthPasswordSubmit, 300);
                }
            });
    }, [csrf, dispatch, sendError]);

    const update = useCallback(() => {
        clearTimeout(updateTimeoutRef.current);
        updateTimeoutRef.current = setTimeout(
            () =>
                api
                    .request('/auth/new/magic/status/', {
                        track_id: magicTrack,
                        csrf_token: magicCSRF
                    })
                    .done(({state, errors, status}) => {
                        if (status === 'ok' && state === 'otp_auth_finished') {
                            return clearIntervals();
                        }

                        if (state === 'auth_challenge') {
                            return sendError(state, true);
                        }

                        if (errors) {
                            const error = (Array.isArray(errors) && errors[0]) || 'unknown';

                            if (
                                [
                                    'track.not_found',
                                    'internal.temporary',
                                    'oauth_token.invalid',
                                    'backend.redis_failed',
                                    'oauth_code.not_found'
                                ].includes(error)
                            ) {
                                clearIntervals();
                                reAuthPasswordSubmit();
                            } else {
                                sendError(error, Boolean(state));
                            }
                        }

                        return update();
                    })
                    .fail(() => {
                        if (timeoutTimeoutRef.current > 1) {
                            sendError('global');
                        } else {
                            ++timeoutTimeoutRef.current;
                            update();
                        }
                    }),
            updateDelayRef.current
        );
    }, [magicCSRF, magicTrack, sendError, reAuthPasswordSubmit, clearIntervals]);

    useEffect(() => {
        if (!active || isLoading) {
            return;
        }

        if (magicTrack && magicCSRF) {
            clearIntervals();
            reset();
            update();
        } else {
            reAuthPasswordSubmit();
        }
    }, [reset, update, active, clearIntervals, isLoading, magicCSRF, magicTrack, reAuthPasswordSubmit]);

    useEffect(() => clearIntervals, [clearIntervals]);

    return {
        magicCSRF,
        magicTrack,
        isLoading: isLoading || !magicTrack || !magicCSRF
    };
};
