import React, {memo, useState} from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import {Input} from '@components/Input';
import {RadioButton} from '@components/RadioButton';
import {Icon} from '@components/IconLego';
import {DeleteDataButtonIconArrowDown} from '../Button/ArrowDown';
import {ServiceItem} from '../Service';

import './Content.styl';

const b = cn('DeleteDataContent');

const MAX_ITEMS_AMOUNT = 5;

const filterItems = (items, filter) =>
    !filter
        ? items
        : items.reduce((acc, cur) => {
              const {key, name} = cur;

              if ([key, name].some((item) => item.toLowerCase().includes(filter))) {
                  acc.push(cur);
              }

              return acc;
          }, []);

export const DeleteDataContent = memo(({itemsWithData, businessItemsWithData, itemsWithoutData}) => {
    const [emptyVisible, setEmptyVisibility] = useState(false);
    const [moreVisible, setMoreVisibility] = useState(false);
    const [serviceType, setServiceType] = useState('all');
    const [filter, setFilter] = useState('');
    const lowerCaseFilter = filter.toLowerCase();
    const allServicesWithData = filterItems(itemsWithData, lowerCaseFilter);
    const servicesTabMap = {
        all: allServicesWithData,
        business: businessItemsWithData
    };
    const servicesWithoutData = filterItems(itemsWithoutData, lowerCaseFilter);
    const services = filter
        ? [allServicesWithData, servicesWithoutData]
        : [servicesTabMap[serviceType], servicesWithoutData];

    return (
        <div className={b('inner')}>
            {(filter || allServicesWithData.length != 0) && (
                <div className={b('filter')}>
                    <Input
                        view='thin-input'
                        value={filter}
                        size='l'
                        placeholder={i18n('DeleteData.services.find')}
                        onChange={(e) => {
                            setFilter(e.target.value);
                            setMoreVisibility(false);
                        }}
                        iconRight={
                            <Icon
                                glyph={filter ? 'type-cross-websearch' : undefined}
                                onClick={() => {
                                    setFilter('');
                                }}
                            />
                        }
                    />
                </div>
            )}
            {!filter && allServicesWithData.length != 0 && (
                <div className={b('tabs')}>
                    <RadioButton
                        width='max'
                        value={serviceType}
                        view='rounded'
                        options={[
                            {value: 'all', children: i18n('DeleteData.services.all')},
                            {value: 'business', children: i18n('DeleteData.services.for-business')}
                        ]}
                        onChange={(event) => {
                            if (serviceType !== event.target.value) {
                                setMoreVisibility(false);
                                setServiceType(event.target.value);
                            }
                        }}
                    />
                </div>
            )}
            <div className={b('content')}>
                {services.map((items, index) => {
                    const noData = index === 1;
                    const hasData = index === 0 && !filter;

                    if (noData && !items.length) {
                        return null;
                    }

                    const hideItemsMoreMax = hasData && !filter;

                    return (
                        <React.Fragment key={items.map(({key}) => key).join('')}>
                            {!filter && noData && (
                                <button className={b('toggle')} onClick={() => setEmptyVisibility(!emptyVisible)}>
                                    <div className={b('toggleLeftPart')}>
                                        <div className={b('toggleIcon', {showed: emptyVisible})}>
                                            <DeleteDataButtonIconArrowDown />
                                        </div>
                                        <div className={b('toggleText')}>{i18n('DeleteData.services.empty')}</div>
                                    </div>
                                    <div className={b('toggleCounter')}>{items.length}</div>
                                </button>
                            )}
                            <div
                                className={b('items', {
                                    active: hasData || emptyVisible || Boolean(filter)
                                })}
                            >
                                {items
                                    .slice(0, hideItemsMoreMax ? MAX_ITEMS_AMOUNT : items.length)
                                    .map(({isGroup, isPredefined, key}) => (
                                        <ServiceItem
                                            key={key}
                                            serviceName={key}
                                            noData={noData}
                                            isGroup={isGroup}
                                            isPredefined={isPredefined}
                                        />
                                    ))}
                                {hideItemsMoreMax && items.length > MAX_ITEMS_AMOUNT && !moreVisible && (
                                    <button className={b('showMore')} onClick={() => setMoreVisibility(true)}>
                                        {`${i18n('DeleteData.services.show-more-amount').replace(
                                            '%amount',
                                            items.length - MAX_ITEMS_AMOUNT
                                        )}`}
                                        <DeleteDataButtonIconArrowDown className={b('showMoreIcon')} />
                                    </button>
                                )}
                                {hideItemsMoreMax &&
                                    moreVisible &&
                                    items
                                        .slice(MAX_ITEMS_AMOUNT)
                                        .map(({isGroup, isPredefined, key}) => (
                                            <ServiceItem
                                                key={key}
                                                serviceName={key}
                                                noData={noData}
                                                isGroup={isGroup}
                                                isPredefined={isPredefined}
                                            />
                                        ))}
                                {hasData && serviceType === 'all' && items.length === 0 && (
                                    <div className={b('noItems')}>
                                        <div className={b('balloonIcon')} />
                                        <div>{i18n('DeleteData.services.no-data-services')}</div>
                                    </div>
                                )}
                                {hasData && serviceType === 'business' && items.length === 0 && (
                                    <div className={b('noItems')}>
                                        <div className={b('balloonIcon')} />
                                        <div>{i18n('DeleteData.services.no-data-business-services')}</div>
                                    </div>
                                )}
                            </div>
                        </React.Fragment>
                    );
                })}
                {filter && services.every((items) => !items.length) && (
                    <div className={b('noItems')}>
                        <div className={b('balloonIcon')} />
                        <div>{i18n('DeleteData.services.no-results')}</div>
                    </div>
                )}
            </div>
        </div>
    );
});

DeleteDataContent.displayName = 'DeleteDataContent';

DeleteDataContent.propTypes = {
    itemsWithData: PropTypes.arrayOf(
        PropTypes.shape({
            key: PropTypes.string.isRequired,
            name: PropTypes.string.isRequired
        })
    ).isRequired,
    businessItemsWithData: PropTypes.arrayOf(
        PropTypes.shape({
            key: PropTypes.string.isRequired,
            name: PropTypes.string.isRequired
        })
    ).isRequired,
    itemsWithoutData: PropTypes.arrayOf(
        PropTypes.shape({
            key: PropTypes.string.isRequired,
            name: PropTypes.string.isRequired
        })
    ).isRequired
};
