import React, {useRef, useMemo, useCallback, useEffect} from 'react';
import {useDispatch, useSelector, shallowEqual} from 'react-redux';
import {cn} from '@bem-react/classname';
import {getData, updateServicesInProgress, checkSession} from './thunks';
import {DeleteDataError} from './Error';
import {DeleteDataContent} from './Content';
import {DeleteDataWhich} from './Which';
import {DeleteDataSep} from './Sep';
import {DeleteDataFAQ} from './FAQ';
import {DeleteDataWarning} from './Warning';
import {DeleteDataTakeout} from './Takeout';
import {DeleteDataSession} from './Session';
import {DeleteDataModalContext} from './ModalContext';
import {DeleteDataServiceSkeleton} from './Service/Skeleton';
import {services, groups, externalServices} from './info';

import './DeleteData.styl';

const b = cn('DeleteData');

const EMPTY_SKELETON_ARRAY = new Array(4).fill(null);
const getPredefinedServices = ({hasMail}) =>
    Object.entries(externalServices).reduce((acc, [key, value]) => {
        if (!value.isHiddenFromList && (key !== 'mail' || hasMail)) {
            acc.push({
                key,
                name: value.name,
                isPredefined: true,
                isBusinessService: value.isBusinessService
            });
        }
        return acc;
    }, []);

const sortData = (arr) => arr.sort((a, b) => (a.name < b.name ? -1 : a.name > b.name ? 1 : 0));
const processData = (data) => sortData(Object.values(data));

export const DeleteData = () => {
    const timeoutRef = useRef(null);
    const {ready, data, servicesInProgress, error, hasMail, isNewLayout, isTakeOutAvailable} = useSelector(
        ({
            header,
            settings: {isNewLayout},
            deleteData: {data, ready, error, servicesInProgress, isTakeOutAvailable}
        }) => ({
            data,
            error,
            ready,
            isNewLayout,
            servicesInProgress,
            hasMail: Boolean(header.defaultAccount.mail),
            isTakeOutAvailable
        }),
        shallowEqual
    );
    const faqContainerRef = useRef();
    const predefinedServices = useMemo(() => getPredefinedServices({hasMail}), [hasMail]);
    const {itemsWithData, businessItemsWithData, itemsWithoutData} = useMemo(() => {
        const {itemsWithData, itemsWithoutData} = Object.entries(data || {}).reduce(
            (acc, [key, value]) => {
                if (key === 'mail' && !hasMail) {
                    return acc;
                }

                const {group, isBusinessService} = services[key];
                const dataKey = value.some(({state}) => state !== 'empty') ? 'itemsWithData' : 'itemsWithoutData';

                if (!group) {
                    acc[dataKey][key] = {key, name: services[key].name, isBusinessService};
                } else if (
                    (!acc.itemsWithData[group] && !acc.itemsWithoutData[group]) ||
                    (dataKey === 'itemsWithData' && acc.itemsWithoutData[group])
                ) {
                    delete acc.itemsWithoutData[group];
                    acc[dataKey][group] = {key: group, name: groups[group].name, isGroup: true};
                }

                return acc;
            },
            {
                itemsWithData: {...predefinedServices},
                itemsWithoutData: {}
            }
        );

        return {
            itemsWithData: processData(itemsWithData),
            itemsWithoutData: processData(itemsWithoutData),
            businessItemsWithData: processData(itemsWithData).filter(({isBusinessService}) => isBusinessService)
        };
    }, [data, hasMail, predefinedServices]);
    const onScrollToFaq = useCallback(() => faqContainerRef.current.scrollIntoView({behavior: 'smooth'}), []);
    const dispatch = useDispatch();

    useEffect(() => {
        if (!ready) {
            dispatch(getData());
        }
    }, [ready, dispatch]);
    useEffect(() => {
        const interval = setInterval(() => dispatch(checkSession()), 60000);

        dispatch(updateServicesInProgress());
        return () => clearInterval(interval);
    }, [dispatch]);
    useEffect(() => {
        if ((servicesInProgress || []).length) {
            timeoutRef.current = setInterval(() => dispatch(getData(servicesInProgress)), 30000);
            return () => clearInterval(timeoutRef.current);
        }
    }, [dispatch, servicesInProgress]);

    return (
        <DeleteDataModalContext>
            <div className={b({nl: isNewLayout})}>
                <div className={b('block', {main: true, nl: isNewLayout})}>
                    <h1 className={b('title')}>{i18n('DeleteData.main.title')}</h1>
                    <div className={b('dude')} />
                    <div className={b('desc')}>{i18n('DeleteData.main.description')}</div>
                    <div className={b('desc', {p2: true})}>{i18n('DeleteData.main.description2')}</div>
                    {isTakeOutAvailable ? (
                        <div className={b('warning')}>
                            <DeleteDataTakeout className={b('takeout')} />
                            <div className={b('infoDrawer')}>
                                <DeleteDataWhich />
                            </div>
                            <div className={b('infoDrawer')}>
                                <DeleteDataSep sepOpenBtnIcon='info' sepOpenBtnText={i18n('DeleteData.sep.title')} />
                            </div>
                        </div>
                    ) : (
                        <DeleteDataWarning decor='download' text={i18n('DeleteData.warning.takeout.text')}>
                            <DeleteDataSep sepOpenBtnText={i18n('DeleteData.warning.takeout.sepOpenBtnText')} />
                        </DeleteDataWarning>
                    )}
                </div>
                <div className={b('block', {nl: isNewLayout})}>
                    <div className={b('blockTitle')}>{i18n('DeleteData.secondary.title')}</div>
                    <div className={b('blockDesc')}>{i18n('DeleteData.secondary.desc')}</div>
                    <div className={b('warning', {marginBottom: true})}>
                        <DeleteDataWarning
                            decor='info'
                            text={i18n('DeleteData.warning.desc')}
                            buttonText={i18n('DeleteData.know-more')}
                            onClick={onScrollToFaq}
                        />
                    </div>
                    {ready && (
                        <DeleteDataContent
                            itemsWithData={itemsWithData}
                            itemsWithoutData={itemsWithoutData}
                            businessItemsWithData={businessItemsWithData}
                        />
                    )}
                    {!error &&
                        !ready &&
                        EMPTY_SKELETON_ARRAY.map((_, index) => <DeleteDataServiceSkeleton key={index} />)}
                    {error && <DeleteDataError />}
                </div>
                <div ref={faqContainerRef} className={b('block', {faq: true, nl: isNewLayout})}>
                    <div className={b('blockTitle')}>{i18n('DeleteData.faq.title')}</div>
                    <DeleteDataFAQ />
                </div>
            </div>
            <DeleteDataSession />
        </DeleteDataModalContext>
    );
};
