import React, {useRef, useEffect} from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import {Button} from '@components/Button';
import {DeleteDataDrawer} from '../../Drawer';

import './Modal.styl';
import {shallowEqual, useSelector} from 'react-redux';
import {CUSTOM_POPUP_ERRORS} from '../../constants';

const b = cn('DeleteDataServiceModal');

const DESCRIPTIONS = {
    // zen: i18n('DeleteData.service.zen.deletion.description'),
    alisa: i18n('DeleteData.service.alice.deletion.description'),
    carsharing: i18n('DeleteData.service.carsharing.deletion.description'),
    eats: i18n('DeleteData.service.eats.deletion.description'),
    lavka: i18n('DeleteData.service.lavka.deletion.description'),
    maps: i18n('DeleteData.service.maps.deletion.description'),
    mapsmrc: i18n('DeleteData.service.mapsmrc.deletion.description'),
    mapspano: i18n('DeleteData.service.mapspano.deletion.description'),
    mapsugc: i18n('DeleteData.service.mapsugc.deletion.description'),
    music: i18n('DeleteData.service.music.deletion.description'),
    nmaps: i18n('DeleteData.service.nmaps.deletion.description'),
    ugc: i18n('DeleteData.service.ugc.deletion.description'),
    weather: i18n('DeleteData.service.weather.deletion.description')
};
const DESCRIPTIONS_CONDITIONAL_ITEMS = {
    music: {RECOMMENDATIONS: i18n('DeleteData.service.music.deletion.description')},
    alisa: {
        Logs: i18n('DeleteData.service.alice.deletion.description'),
        Contacts: i18n('DeleteData.service.alice.contacts.deletion.description')
    }
};

const shouldShowCustomDescription = (serviceName, itemsToDelete) =>
    Array.isArray(itemsToDelete) &&
    (!DESCRIPTIONS_CONDITIONAL_ITEMS[serviceName] ||
        itemsToDelete.some((item) => DESCRIPTIONS_CONDITIONAL_ITEMS[serviceName][item]));

const getDescription = (serviceName, itemsToDelete, customErrorPopup) => {
    if (customErrorPopup) {
        const {error, service} = customErrorPopup;

        if (error && service) {
            const {description} = CUSTOM_POPUP_ERRORS[service][error] || CUSTOM_POPUP_ERRORS[service].default;

            if (description) {
                return description;
            }
        }
    }
    if (
        shouldShowCustomDescription(serviceName, itemsToDelete) &&
        (DESCRIPTIONS[serviceName] || DESCRIPTIONS_CONDITIONAL_ITEMS[serviceName])
    ) {
        return (
            (DESCRIPTIONS_CONDITIONAL_ITEMS[serviceName] &&
                DESCRIPTIONS_CONDITIONAL_ITEMS[serviceName][itemsToDelete[0]]) ||
            DESCRIPTIONS[serviceName]
        );
    }

    return i18n('DeleteData.deletion.description');
};

const getActionButtonText = (customErrorPopup) => {
    if (customErrorPopup) {
        const {error, service} = customErrorPopup;
        const {actionButtonText} = CUSTOM_POPUP_ERRORS[service][error] || CUSTOM_POPUP_ERRORS[service].default;

        if (actionButtonText) {
            return actionButtonText;
        }
    }

    return i18n('DeleteData.delete');
};

export const DeleteDataServiceModal = ({visible, serviceName, onClose, onConfirm, itemsToDelete}) => {
    const lastItemsToDelete = useRef(itemsToDelete);

    useEffect(() => {
        if (itemsToDelete) {
            lastItemsToDelete.current = itemsToDelete;
        }
    }, [itemsToDelete]);

    const {customErrorPopup} = useSelector(
        ({deleteData: {customErrorPopup} = {}} = {}) => ({customErrorPopup}),
        shallowEqual
    );

    const hideCancelButton = Boolean(
        customErrorPopup &&
            (CUSTOM_POPUP_ERRORS[customErrorPopup.service][customErrorPopup.error] ||
                CUSTOM_POPUP_ERRORS[customErrorPopup.service].default)
    );

    return (
        <DeleteDataDrawer visible={visible} onClose={onClose}>
            <div className={b('desc')}>
                {getDescription(serviceName, itemsToDelete || lastItemsToDelete.current, customErrorPopup)}
            </div>
            <div className={b('actions')}>
                <Button view='action' width='max' size='l' onClick={onConfirm} className={b('action')}>
                    {getActionButtonText(customErrorPopup)}
                </Button>
                {!hideCancelButton && (
                    <Button view='default' size='l' width='max' onClick={onClose} className={b('action')}>
                        {i18n('DeleteData.cancel')}
                    </Button>
                )}
            </div>
        </DeleteDataDrawer>
    );
};

DeleteDataServiceModal.propTypes = {
    visible: PropTypes.bool,
    onClose: PropTypes.func.isRequired,
    onConfirm: PropTypes.func.isRequired,
    serviceName: PropTypes.string.isRequired,
    itemsToDelete: PropTypes.arrayOf(PropTypes.string)
};
