import React, {memo, useState, useMemo, useCallback} from 'react';
import {useDispatch, useSelector, shallowEqual} from 'react-redux';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import metrics from '@blocks/metrics';
import {DELETION_STATE, DELETE_DATA_GOAL_PREFIX} from '../constants';
import {services} from '../info';
import {processServiceItems} from '../util';
import {deleteServiceItems, hideCustomError} from '../thunks';
import {DeleteDataCard} from '../Card';
import {DeleteDataServiceItem} from './Item';
import {DeleteDataServiceModal} from './Modal';
import {Spin} from '@components/Spin';
import {DeleteDataUnbind} from '../Unbind';
import {DeleteDataServiceTrachCan} from '../TrashCan';

import './Service.styl';

const b = cn('DeleteDataService');
const sendDeletionMetrics = (title, serviceName, items) =>
    metrics.send([
        title,
        services[serviceName].name,
        items.length > 1 ? 'Все данные' : services[serviceName].fields[items[0]].title
    ]);
const goalDeleteServiceMetrics = (serviceName) =>
    metrics.goal(`${DELETE_DATA_GOAL_PREFIX}_delete_service_${serviceName}`);

export const DeleteDataService = memo(({serviceName}) => {
    const dispatch = useDispatch();
    const {hasError, data, loadingServiceItems, showCustomErrorPopup} = useSelector(
        ({deleteData: {data, loadingServiceItems, errorServiceItems, customErrorPopup}}) => ({
            data: data[serviceName],
            loadingServiceItems: loadingServiceItems[serviceName],
            showCustomErrorPopup: Boolean(customErrorPopup && customErrorPopup.service === serviceName),
            hasError: Object.values(errorServiceItems[serviceName] || {}).some(Boolean)
        }),
        shallowEqual
    );
    const {isTesting} = useSelector(
        ({settings: {env: {type: envType} = {}} = {}}) => ({
            isTesting: ['development', 'testing'].includes(envType)
        }),
        shallowEqual
    );
    const deletingAll =
        Boolean(loadingServiceItems) &&
        data.some(({state}) => state === DELETION_STATE.ready) &&
        data.every(({id, state}) => state !== DELETION_STATE.ready || loadingServiceItems[id]);
    const deletableItems = useMemo(
        () =>
            (data || [])
                .filter(({id, state}) => !(loadingServiceItems || {})[id] && state === DELETION_STATE.ready)
                .map(({id}) => id),
        [data, loadingServiceItems]
    );
    const [itemsToDelete, setItemsToDelete] = useState(null);
    const {name} = services[serviceName];
    const onClose = useCallback(() => {
        if (showCustomErrorPopup) {
            dispatch(hideCustomError());
        } else {
            sendDeletionMetrics('Отмена удаления', serviceName, itemsToDelete);
        }
        setItemsToDelete(null);
    }, [dispatch, serviceName, itemsToDelete, showCustomErrorPopup]);
    const onDeleteByLink = useCallback(() => {
        metrics.send(['Переход по ссылке удаления', name]);
        metrics.goal(`${DELETE_DATA_GOAL_PREFIX}_delete_service_${serviceName}`);
    }, [name, serviceName]);
    const onDeletionAttempt = useCallback((ids) => setItemsToDelete([].concat(ids)), []);
    const onDeleteAll = useCallback(() => setItemsToDelete(deletableItems), [deletableItems]);
    const onConfirm = useCallback(() => {
        if (showCustomErrorPopup) {
            dispatch(hideCustomError());
        } else {
            sendDeletionMetrics('Удаление', serviceName, itemsToDelete);
            goalDeleteServiceMetrics(serviceName);
            dispatch(deleteServiceItems(serviceName, itemsToDelete));
        }
        setItemsToDelete(null);
    }, [dispatch, itemsToDelete, serviceName, showCustomErrorPopup]);
    const processedData = processServiceItems(serviceName, data);
    const hasDeleteAll = (() => {
        if (processedData && processedData.fields.length > 1) {
            let found = 0;

            for (const {state} of processedData.fields) {
                if (state === DELETION_STATE.ready) {
                    found++;
                    if (found > 1) {
                        return true;
                    }
                }
            }
        }
        return false;
    })();

    return (
        processedData && (
            <>
                <DeleteDataCard
                    hasError={hasError}
                    loading={deletingAll}
                    serviceName={serviceName}
                    title={processedData.name}
                    description={processedData.description}
                    hasInProgress={processedData.fields.some(({state}) => state === DELETION_STATE.inProgress)}
                >
                    {services[serviceName].unbind && <DeleteDataUnbind className={b('unbind')} />}
                    {processedData.fields.map((field) => (
                        <DeleteDataServiceItem
                            deleteLink={
                                field.hasLink && field.links ? (isTesting ? field.links.test : field.links.prod) : ''
                            }
                            key={field.id}
                            id={field.id}
                            state={field.state}
                            title={field.title}
                            serviceName={serviceName}
                            onDelete={field.hasLink ? onDeleteByLink : onDeletionAttempt}
                            description={field.description}
                            updateDate={field.update_date}
                        />
                    ))}
                    {hasDeleteAll && (
                        <>
                            <div className={b('sep')} />
                            <button disabled={deletingAll} className={b('delete')} onClick={onDeleteAll}>
                                <div className={b('deleteAction')}>
                                    <Spin progress={deletingAll} size='xxs' />
                                    {!deletingAll && <DeleteDataServiceTrachCan />}
                                </div>
                                <div className={b('deleteTitle')}>{i18n('DeleteData.delete.all')}</div>
                            </button>
                        </>
                    )}
                </DeleteDataCard>
                <DeleteDataServiceModal
                    onConfirm={onConfirm}
                    serviceName={serviceName}
                    showCustomErrorPopup={showCustomErrorPopup}
                    onClose={onClose}
                    itemsToDelete={itemsToDelete}
                    visible={Boolean(itemsToDelete || showCustomErrorPopup)}
                />
            </>
        )
    );
});

DeleteDataService.displayName = 'DeleteDataService';

DeleteDataService.propTypes = {
    serviceName: PropTypes.string.isRequired
};
