import React, {memo, useState, useCallback} from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import {useDispatch, useSelector, shallowEqual} from 'react-redux';
import metrics from '@blocks/metrics';
import getArchiveData from '@blocks/takeout/actions/getArchiveData';
import copyToClipboard from '@blocks/takeout/actions/copyToClipboard';
import requestDataArchive from '@blocks/takeout/actions/requestDataArchive';
import getArchivePassword from '@blocks/takeout/actions/getArchivePassword';
import {Spin} from '@components/Spin';
import {Button} from '@components/Button';
import {DeleteDataDrawer} from '../Drawer';
import {DeleteDataToaster} from '../Toaster';
import {DeleteDataTakeoutCopyIcon} from './CopyIcon';
import {DeleteDataTakeoutConfirmation} from './Confirmation';
import {DeleteDataTakeoutDownloadIcon} from './DownloadIcon';
import {DELETE_DATA_GOAL_PREFIX} from '../constants';

import './Takeout.styl';

const b = cn('DeleteDataTakeout');

const ZIP_LINK = 'https://www.7-zip.org/download.html';
const ZIP_LINK_MAC = 'https://theunarchiver.com/';
const ZIP_PROG_NAME = '7-zip';
const ZIP_PROG_NAME_MAC = 'The Unarchiver';

const sendMetrics = (text) => metrics.send(['Архив', text]);
const goalMetrics = (text) => metrics.goal(`${DELETE_DATA_GOAL_PREFIX}_${text}`);

export const DeleteDataTakeout = memo(({className}) => {
    const dispatch = useDispatch();
    const {
        takeOutInfo: {
            error,
            status,
            copyError,
            isLoading,
            isLinkLoading,
            isUnavailable,
            archivePassword,
            archiveInfo = {},
            hasPasswordCopied,
            isPasswordLoading
        },
        settings: {isMac},
        defaultAccount: {displayName, displayLogin, login}
    } = useSelector(
        ({header, settings, takeOutInfo}) => ({
            settings,
            takeOutInfo,
            defaultAccount: header.defaultAccount
        }),
        shallowEqual
    );
    const [isModalVisible, setModalVisibility] = useState(false);
    const onOpen = useCallback(() => setModalVisibility(true), []);
    const onDownload = useCallback(() => {
        sendMetrics('Запрос на скачивание');
        goalMetrics('archive_download');
        dispatch(getArchiveData());
    }, [dispatch]);
    const onCopy = useCallback(() => {
        sendMetrics('Скопировать пароль');
        dispatch(copyToClipboard({elementId: 'copied-text'}));
    }, [dispatch]);
    const onClose = useCallback(() => setModalVisibility(false), []);
    const onPasswordRequest = useCallback(() => {
        sendMetrics('Запрос пароля');
        dispatch(getArchivePassword());
    }, [dispatch]);
    const onConfirm = useCallback(() => {
        sendMetrics('Запрос на создание');
        goalMetrics('archive_order');
        dispatch(requestDataArchive());
    }, [dispatch]);
    const isInProgress = status === 'isBeingRequesting';
    const isReady = status === 'isReady';
    const toasterText = (() => {
        if (hasPasswordCopied) {
            return i18n('TakeOut.getdata_pwd_copied');
        }

        return copyError ? i18n('TakeOut.getdata_pwd_copied-error') : null;
    })();

    return (
        !isUnavailable && (
            <div className={b('', [className])}>
                {isInProgress || isReady ? (
                    <>
                        <div className={b('info')}>
                            <div className={b('infoIcon', {ready: isReady})} />
                            <div className={b('infoDesc')}>
                                {isReady ? (
                                    <>
                                        <div className={b('infoPrepared')}>
                                            {i18n('DeleteData.takeout.prepared.title')}
                                        </div>
                                        <div
                                            className={b('infoZipDownload')}
                                            dangerouslySetInnerHTML={{
                                                __html: i18n('DeleteData.takeout.prepared.download')
                                                    .replace('%zip-href%', isMac ? ZIP_LINK_MAC : ZIP_LINK)
                                                    .replace('%prog-name%', isMac ? ZIP_PROG_NAME_MAC : ZIP_PROG_NAME)
                                            }}
                                        />
                                        {archiveInfo.valid && (
                                            <div className={b('infoValid')}>
                                                {i18n('TakeOut.getdata_available_till')} {archiveInfo.valid}
                                            </div>
                                        )}
                                    </>
                                ) : (
                                    <div>{i18n('DeleteData.takeout.preparing')}</div>
                                )}
                            </div>

                            {isReady && (
                                <div className={b('infoActions')}>
                                    {archiveInfo.isPasswordRequired && (
                                        <div className={b('infoActionsItem')}>
                                            <Button
                                                size='l'
                                                view='default'
                                                width='max'
                                                onClick={archivePassword ? onCopy : onPasswordRequest}
                                                addonAfter={
                                                    <Spin
                                                        size='xxs'
                                                        view='default'
                                                        className={b('buttonSpin')}
                                                        progress={!archivePassword && isPasswordLoading}
                                                    />
                                                }
                                            >
                                                {archivePassword ? (
                                                    <>
                                                        <DeleteDataTakeoutCopyIcon className={b('copyIcon')} />
                                                        <span>{archivePassword}</span>
                                                    </>
                                                ) : (
                                                    i18n('DeleteData.show-password')
                                                )}
                                            </Button>
                                        </div>
                                    )}
                                    <div className={b('infoActionsItem')}>
                                        <Button
                                            size='l'
                                            width='max'
                                            view='action'
                                            onClick={onDownload}
                                            progress={isLinkLoading}
                                        >
                                            {i18n('DeleteData.takeout.download.archive')}
                                        </Button>
                                    </div>
                                </div>
                            )}
                        </div>
                        {error && <div className={b('infoError')}>{error}</div>}
                        {isReady && (
                            <>
                                <div className={b('create')}>
                                    <Button
                                        size='l'
                                        view='pseudo'
                                        width='max'
                                        onClick={onConfirm}
                                        addonAfter={
                                            <Spin
                                                size='xxs'
                                                view='default'
                                                className={b('buttonSpin')}
                                                progress={isLoading && !isPasswordLoading && !isLinkLoading}
                                            />
                                        }
                                    >
                                        {i18n('DeleteData.create')}
                                    </Button>
                                </div>
                                <input readOnly='readOnly' id='copied-text' value={archivePassword} />
                            </>
                        )}
                    </>
                ) : (
                    <>
                        <div className={b('actions')}>
                            <Button view='action' size='l' onClick={onOpen} className={b('action')} width='max'>
                                <DeleteDataTakeoutDownloadIcon className={b('icon')} />
                                {i18n('DeleteData.takeout.download')}
                            </Button>
                        </div>
                        <DeleteDataDrawer
                            visible={isModalVisible}
                            onClose={isLoading ? undefined : onClose}
                            title={`${i18n('DeleteData.takeout.confirmation.title')} ${displayName ||
                                displayLogin ||
                                login}`}
                        >
                            {error && <div className={b('errorSign')} />}
                            <DeleteDataTakeoutConfirmation
                                onCancel={onClose}
                                onConfirm={onConfirm}
                                isLoading={isLoading}
                            />
                        </DeleteDataDrawer>
                    </>
                )}
                <DeleteDataToaster icon='ok' text={toasterText} />
            </div>
        )
    );
});

DeleteDataTakeout.displayName = 'DeleteDataTakeout';

DeleteDataTakeout.propTypes = {
    className: PropTypes.string
};
