import api from '@blocks/api';
import {
    setData,
    setError,
    mergeData,
    setReadiness,
    showReloadDrawer,
    mergeServiceData,
    setServiceItemsError,
    clearServiceItemsError,
    setServiceItemsLoading,
    setServicesInProgress,
    showCustomErrorPopup,
    hideCustomErrorPopup
} from './actions';
import {getQuery} from '@blocks/utils';
import {DELETION_STATE, CUSTOM_POPUP_ERRORS} from './constants';

const getExtendedAttrs = () => {
    const query = getQuery();

    return {
        mimino: query.mimino === '0' ? 0 : undefined,
        skip_sess_check: query.skip_sess_check === '1' ? 1 : undefined,
        sess_lifetime: Number(query.sess_lifetime || 0) || undefined
    };
};

const isSessionError = (err) =>
    Array.isArray((err || {}).errors) && ['csrf_token.invalid', 'session.invalid'].includes(err.errors[0]);

const isCustomPopupError = ({errors: [{code} = {}] = []} = {}, service) =>
    Boolean(
        code &&
            CUSTOM_POPUP_ERRORS[service] &&
            (CUSTOM_POPUP_ERRORS[service][code] || CUSTOM_POPUP_ERRORS[service].default)
    );

export const updateServicesInProgress = () => (dispatch, getState) => {
    const {data, servicesInProgress = []} = getState().deleteData;

    if (data) {
        const nextItems = Object.entries(data).reduce((acc, [key, value]) => {
            if (value.some((item) => item.state === DELETION_STATE.inProgress)) {
                acc.push(key);
            }

            return acc;
        }, []);

        if (
            nextItems.length !== servicesInProgress.length ||
            !nextItems.every((item) => servicesInProgress.includes(item))
        ) {
            dispatch(setServicesInProgress(nextItems));
        }
    }
};

export const handleError = (err) => (dispatch) =>
    isSessionError(err) ? dispatch(showReloadDrawer()) : dispatch(setError(err));

export const getData = (() => {
    let isLoading = false;

    return (services) => (dispatch, getState) => {
        const {error, ready, isReloadDrawerVisible} = getState().deleteData;

        if (isReloadDrawerVisible || isLoading || (error && services)) {
            return;
        }

        dispatch(setError(null));
        isLoading = true;

        return api
            .request('/profile/data/getdata', {services, ...getExtendedAttrs()})
            .done(({status, data}) => {
                if (status === 'ok' && data) {
                    dispatch((services ? mergeData : setData)(data));
                    dispatch(updateServicesInProgress());
                    if (!ready) {
                        dispatch(setReadiness(true));
                    }
                } else if (status === 'error' && !services) {
                    dispatch(setError(new Error()));
                }
            })
            .fail((err) => (isSessionError(err) ? dispatch(showReloadDrawer()) : !services && dispatch(setError(err))))
            .always(() => {
                isLoading = false;
            });
    };
})();

export const addServiceInProgress = (service) => (dispatch, getState) => {
    const {servicesInProgress = []} = getState().deleteData;

    if (!servicesInProgress.includes(service)) {
        dispatch(setServicesInProgress([...servicesInProgress, service]));
    }
};

export const checkSession = () => (dispatch) =>
    api
        .request('/profile/data/check-session', getExtendedAttrs())
        .fail((err = {}) => isSessionError(err) && dispatch(showReloadDrawer()));

export const deleteServiceItems = (service, ids) => (dispatch, getState) => {
    const {isReloadDrawerVisible} = getState().deleteData;
    const onError = (errors) => {
        dispatch(setServiceItemsError(service, ids, (Array.isArray(errors) && errors[0]) || {code: 'custom'}));
    };

    if (isReloadDrawerVisible) {
        return;
    }

    dispatch(setServiceItemsLoading(service, ids, true));
    dispatch(clearServiceItemsError(service));

    // return setTimeout(() => {
    //     dispatch(
    //         mergeServiceData({
    //             [service]: ids.reduce(
    //                 (acc, cur) => ({
    //                     ...acc,
    //                     [cur]: {
    //                         state: DELETION_STATE.inProgress
    //                     }
    //                 }),
    //                 {}
    //             )
    //         })
    //     );
    //     dispatch(addServiceInProgress(service));
    //     dispatch(setServiceItemsLoading(service, ids, false));
    // }, 10000);

    return api
        .request('/profile/data/delete', {service, id: ids, ...getExtendedAttrs()})
        .done(({status, errors}) => {
            if (status === 'ok') {
                dispatch(
                    mergeServiceData({
                        [service]: ids.reduce(
                            (acc, cur) => ({
                                ...acc,
                                [cur]: {
                                    state: DELETION_STATE.inProgress
                                }
                            }),
                            {}
                        )
                    })
                );
                dispatch(addServiceInProgress(service));
            } else {
                onError(errors);
            }
        })
        .fail((err = {}) => {
            if (isSessionError(err)) {
                return dispatch(showReloadDrawer());
            }
            if (isCustomPopupError(err, service)) {
                return dispatch(showCustomErrorPopup({error: err.errors[0].code, service}));
            }
            return onError(err.errors);
        })
        .always(() => dispatch(setServiceItemsLoading(service, ids, false)));
};

export const hideCustomError = () => (dispatch) => dispatch(hideCustomErrorPopup());
