import './PhoneConfirmation.styl';
import React from 'react';
import PropTypes from 'prop-types';
import {Button} from '@components/Button';
import {Title} from '@components/Title';
import {BackButton} from '@components/BackButton';
import {Field} from '@components/Field';
import {FIELDS_NAMES} from '@components/Field/names';
import PhoneConfirmationTimer from '@blocks/authv2/components/PhoneConfirmationTimer/PhoneConfirmationTimer.jsx';
import {classNames, getUnixTimeStamp} from '@blocks/utils';
import {MODE_BROWSER} from '@blocks/SignUp/modes/constants';
import metrika from '@blocks/metrics';
import {Password} from '@components/Password';
import {CodeField} from '@components/CodeField';

const b = classNames('PhoneConfirmationCode');

class PhoneConfirmation extends React.PureComponent {
    state = {
        currentTimestamp: getUnixTimeStamp(),
        code: []
    };

    static defaultProps = {
        name: FIELDS_NAMES.PHONE_CODE
    };

    ref = React.createRef();
    isAutoconfirmSend = false;

    _timerInterval = null;

    componentDidMount() {
        const {confirmMethod} = this.props;

        this._stopTimer();
        this._startTimer();

        metrika.send('Показ экрана потдверждения телефона');
        metrika.send(`Метод подтверждения телефона - ${confirmMethod}`);

        if (this.ref.current) {
            this.ref.current.focus();
        }

        this.isAutoconfirmSend = false;
    }

    componentDidUpdate(prevProps) {
        if (this.props.denyResendUntil !== prevProps.denyResendUntil) {
            this._updateTimestamp();
            this._startTimer();
        }

        if (this.props.confirmMethod !== prevProps.confirmMethod) {
            metrika.send(`Метод подтверждения телефона - ${this.props.confirmMethod}`);
        }

        if (this.props.code.length === this._getCodeLength() && this.props.isAm && !this.isAutoconfirmSend) {
            metrika.send('Введен код подтверждения телефона');
            this._onConfirm();
        }

        if (prevProps.error && !this.props.error && this.isAutoconfirmSend) {
            this.isAutoconfirmSend = false;
        }
    }

    componentWillUnmount() {
        const {updateValues} = this.props;

        this.clearErrors();
        this._stopTimer();
        updateValues({field: FIELDS_NAMES.PHONE_CODE, value: ''});
    }

    _updateTimestamp = () => this.setState({currentTimestamp: getUnixTimeStamp()});

    _startTimer = () => {
        const {denyResendUntil} = this.props;

        if (!denyResendUntil) {
            return;
        }

        this._timerInterval = setInterval(() => {
            this._updateTimestamp();

            if (getUnixTimeStamp() >= Number(denyResendUntil)) {
                this._stopTimer();
            }
        }, 1000);
    };

    _stopTimer = () => {
        if (this._timerInterval) {
            clearInterval(this._timerInterval);
            this._timerInterval = null;
        }
    };

    _getCodeLength = () => {
        const {confirmMethod, isWithCodeFormatExp} = this.props;

        return confirmMethod === 'by_flash_call' ? 4 : isWithCodeFormatExp ? 7 : 6;
    };

    _getSendSMSButtonText = (showTimer) => {
        const {confirmMethod, denyResendUntil} = this.props;
        const {currentTimestamp} = this.state;
        const confirmByCall = ['by_flash_call', 'by_call'].includes(confirmMethod);
        const text = confirmByCall
            ? i18n('_AUTH_.registration.btn.getsms')
            : i18n('_AUTH_.enter-confirmation-code.send-again.v2');

        if (showTimer) {
            return (
                <>
                    {text}
                    {'  '}
                    <PhoneConfirmationTimer
                        startTime={currentTimestamp}
                        endTime={Number(denyResendUntil)}
                        isUnixTimeStamp={true}
                    />
                </>
            );
        }

        return text;
    };

    _getTitle = () => {
        const {confirmMethod, phone, shouldCheckPhone} = this.props;

        if (shouldCheckPhone) {
            return i18n('_AUTH_.phoneconfirmation.title.callfail');
        }

        switch (confirmMethod) {
            case 'by_flash_call': {
                return i18n('_AUTH_.phoneconfirmation.title.flashcall');
            }

            case 'by_call': {
                return i18n('_AUTH_.phoneconfirmation.title.call');
            }

            default: {
                return i18n('_AUTH_.phoneconfirmation.title.sms', phone);
            }
        }
    };

    _retryToRequestPhoneConfirmationCode = () => {
        const {onRetry} = this.props;

        this._clearCode();
        metrika.send('Отправка СМС');
        onRetry();
    };

    _clearCode = () => {
        this.setState({code: []});
        const nodes = document.querySelectorAll(`.${b('input')}`);
        const inputs = nodes && nodes.length ? Array.prototype.slice.call(nodes) : [];

        inputs.forEach((input = {}) => {
            input.value = '';
        });

        if (this.ref.current) {
            this.ref.current.focus();
        }
    };

    clearErrors = () => {
        const {name, updateErrors} = this.props;

        updateErrors({field: name, error: {}});
    };

    getButtonSize = () => {
        const {place} = this.props;

        if (place === MODE_BROWSER) {
            return 'xm';
        }

        return 'l';
    };

    getFieldView = () => {
        const {place} = this.props;

        if (place === MODE_BROWSER) {
            return 'one-border';
        }

        return 'big-input';
    };

    getFieldSize = () => {
        const {place} = this.props;

        if (place === MODE_BROWSER) {
            return 'm';
        }

        return 'l';
    };

    getPlaceholder = () => {
        const {confirmMethod} = this.props;

        return confirmMethod === 'by_flash_call' ? '1234' : '123 456';
    };

    _onConfirm = () => {
        const {onConfirm} = this.props;

        if (!this.isAutoconfirmSend) {
            this.isAutoconfirmSend = true;
            onConfirm();
        }
    };

    _renderControls = () => {
        const {currentTimestamp} = this.state;
        const {
            denyResendUntil,
            shouldCheckPhone,
            isAutoSubmitPrevent,
            goBack,
            loading,
            onConfirm,
            buttonText,
            buttonView
        } = this.props;
        const showTimer = currentTimestamp < Number(denyResendUntil);

        return (
            <div className={b('controls')}>
                {shouldCheckPhone ? (
                    <>
                        <Button
                            onClick={goBack}
                            text={i18n('_AUTH_.registration.btn.changephone')}
                            view='default'
                            type='button'
                            size={this.getButtonSize()}
                            disabled={loading}
                            width='max'
                            dataT='go-back'
                        />

                        <Button
                            onClick={this._retryToRequestPhoneConfirmationCode}
                            text={i18n('_AUTH_.registration.btn.correctphone')}
                            type='button'
                            view={buttonView}
                            size={this.getButtonSize()}
                            disabled={loading}
                            width='max'
                            dataT='retry-to-request-code'
                        />
                    </>
                ) : (
                    <>
                        <Button
                            text={buttonText || i18n('_AUTH_.next')}
                            onClick={onConfirm}
                            size={this.getButtonSize()}
                            view={buttonView}
                            type={isAutoSubmitPrevent ? 'submit' : 'button'}
                            disabled={this._isDisabled() || loading}
                            width='max'
                        />
                        <Button
                            onClick={this._retryToRequestPhoneConfirmationCode}
                            view='default'
                            type='button'
                            disabled={showTimer || loading}
                            size={this.getButtonSize()}
                            width='max'
                            dataT='retry-to-request-code'
                        >
                            {this._getSendSMSButtonText(showTimer)}
                        </Button>
                    </>
                )}
            </div>
        );
    };

    onFieldChange = () => {
        const {code, confirmMethod, isBindProcess, isAutoSubmitPrevent = false} = this.props;
        const codeFormatted = code.replace(/[^0-9]+/g, '');
        const codeRequiredLength = confirmMethod === 'by_flash_call' ? 4 : 6;

        if (
            codeFormatted.length === codeRequiredLength &&
            !this.isAutoconfirmSend &&
            !isBindProcess &&
            !isAutoSubmitPrevent
        ) {
            metrika.send('Введен код подтверждения телефона');
            this._onConfirm();
        }
    };

    _isDisabled = () => {
        const {error} = this.props;
        const {code} = this.state;

        return code.join('').length === this._getCodeLength() && !error;
    };

    _maybeRenderForm = () => {
        const {
            shouldCheckPhone,
            confirmMethod,
            isMobile,
            isNotProduction,
            trackId,
            isPasswordRequired,
            hasRoundViewExp
        } = this.props;
        const codeRequiredLength = confirmMethod === 'by_flash_call' ? 4 : 6;

        if (shouldCheckPhone) {
            return null;
        }

        if (hasRoundViewExp) {
            return <CodeField codeLength={codeRequiredLength} onKeyUp={this.onFieldChange} />;
        }

        return (
            <>
                <Field
                    name={FIELDS_NAMES.PHONE_CODE}
                    view={this.getFieldView()}
                    size={this.getFieldSize()}
                    disabled={this._isDisabled()}
                    maxLength={this._getCodeLength()}
                    onKeyUp={this.onFieldChange}
                    tInfo={isNotProduction ? `sms_code_input_${trackId}` : undefined}
                    options={{
                        autoComplete: 'off',
                        autoCapitalize: 'off',
                        autoCorrect: 'off',
                        inputMode: isMobile ? 'numeric' : 'text'
                    }}
                    placeholder={this.getPlaceholder()}
                    ref={this.ref}
                    isPhoneCode={true}
                />
                {isPasswordRequired && <Password view='big-input' label={i18n('_AUTH_.security.enter_pwd_label')} />}
            </>
        );
    };

    _maybeRenderPhoneForCheck = () => {
        const {shouldCheckPhone, phone} = this.props;

        if (!shouldCheckPhone) {
            return null;
        }

        return <div className={b('check-phone')}>{phone}</div>;
    };

    _goBack = () => {
        const {goBack, setPhoneConfirmMethod} = this.props;

        goBack();
        setPhoneConfirmMethod('by_sms');
    };

    _maybeRenderBackButton = () => {
        const {goBack, shouldCheckPhone, place} = this.props;

        if (!goBack || shouldCheckPhone || place !== MODE_BROWSER) {
            return null;
        }

        return (
            <div className={b('back-button')}>
                <BackButton onClick={this._goBack} place={place} />
            </div>
        );
    };

    _maybeRenderHead = () => {
        const {place} = this.props;

        if (place !== 'change-password') {
            return (
                <div className={b('head')}>
                    {this._maybeRenderBackButton()}
                    <Title size={this.getTitleSize()}>
                        <div dangerouslySetInnerHTML={{__html: this._getTitle()}} />
                    </Title>
                </div>
            );
        }

        return null;
    };

    getTitleSize = () => {
        const {place} = this.props;

        if (place === MODE_BROWSER) {
            return 'l';
        }
    };

    render() {
        const {place, confirmMethod} = this.props;

        return (
            <div className={b('', {[place]: true})}>
                {this._maybeRenderHead()}
                <div className={b('body', {confirmMethod})}>
                    {this._maybeRenderPhoneForCheck()}
                    {this._maybeRenderForm()}
                </div>

                {this._renderControls()}
            </div>
        );
    }
}

PhoneConfirmation.propTypes = {
    code: PropTypes.string,
    name: PropTypes.string,
    error: PropTypes.string,
    isAm: PropTypes.bool,
    onConfirm: PropTypes.func,
    onRetry: PropTypes.func.isRequired,
    denyResendUntil: PropTypes.oneOfType([PropTypes.number, PropTypes.string]),
    phone: PropTypes.string,
    callingNumberTemplate: PropTypes.string,
    confirmMethod: PropTypes.oneOf(['by_sms', 'by_flash_call', 'by_call']),
    updateValues: PropTypes.func.isRequired,
    updateErrors: PropTypes.func.isRequired,
    goBack: PropTypes.func,
    setPhoneConfirmMethod: PropTypes.func.isRequired,
    place: PropTypes.string,
    shouldCheckPhone: PropTypes.bool,
    isMobile: PropTypes.bool,
    skipTitle: PropTypes.bool,
    loading: PropTypes.bool,
    isWithCodeFormatExp: PropTypes.bool,
    isNotProduction: PropTypes.bool,
    trackId: PropTypes.string,
    isBindProcess: PropTypes.bool,
    buttonText: PropTypes.string,
    buttonView: PropTypes.string,
    isAutoSubmitPrevent: PropTypes.bool,
    isPasswordRequired: PropTypes.bool,
    hasRoundViewExp: PropTypes.bool
};

export {PhoneConfirmation};
