import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {Field} from '@components/Field';
import {FIELDS_NAMES} from '@components/Field/names';
import {Button} from '@components/Button';
import {Password} from '@components/Password';
import {Tumbler} from '@components/Tumbler';
import {TimeLeftTicker} from '@components/TimeLeftTicker';

import {cn} from '@bem-react/classname';

import './PhoneReplace.styl';

const b = cn('PhoneReplace');

export class PhoneReplace extends PureComponent {
    state = {
        showTimer: false
    };

    handlePhoneAvailable = () => {
        const {isOldNumberUnAvailable, togglePhoneAvailable} = this.props;

        togglePhoneAvailable(!isOldNumberUnAvailable);
    };

    getQuarantineMessage = (end, phone) => {
        let text = i18n('Phones.phones.quarantine.replace.many');

        if (end > 1 && end < 5) {
            text = i18n('Phones.phones.quarantine.replace.some');
        }

        if (end <= 1 || end === 21) {
            text = i18n('Phones.phones.quarantine.replace.one');
        }

        return text.replace('%1%2%3', phone).replace('%4', end);
    };

    hideTimer = () => this.setState({showTimer: false});

    showTimer = () => this.setState({showTimer: true});

    sendCode = (event) => {
        const {enteredPhone, hasCodeSent, isInProgress} = this.props;

        if (event) {
            event.preventDefault();
        }

        this.showTimer();

        if (hasCodeSent || isInProgress) {
            this.props.resendCode();
            return;
        }

        this.props.sendCode(enteredPhone);
    };

    renderCodeConfirmation = () => {
        const {processedPhone, availablePhone, isInProgress, isLoading} = this.props;
        const phoneNumberInProgress = processedPhone || availablePhone;

        return (
            <div className={b('codeConfirmation')}>
                <div className={b('description')}>
                    {isInProgress && !processedPhone && <div>{i18n('Phones.phones.init-operation.replace')}</div>}
                    {i18n('ProfileSecurity.security.phone_code-sent').replace('%phone', phoneNumberInProgress)}
                </div>
                <div className={b('enterCode')}>
                    <div className={b('enterCodeInput')}>
                        <Field
                            size='l'
                            placeholder={i18n('ProfileSecurity.security.phone_code-placeholder')}
                            name={FIELDS_NAMES.PHONE_CODE}
                            options={{inputmode: 'numeric'}}
                        />
                    </div>

                    {this.state.showTimer ? (
                        <>
                            <span className={b('sendCodeAgain')}>
                                {i18n('ProfileSecurity.security.ask_code-again')}
                                {'  '}
                                <TimeLeftTicker tickCount={30} onTickEnd={this.hideTimer} />
                            </span>
                        </>
                    ) : (
                        <button
                            data-t='send-code-again-btn'
                            className={b('sendCodeAgainBtn')}
                            type='button'
                            onClick={this.sendCode}
                            disabled={isLoading}
                        >
                            {i18n('ProfileSecurity.security.ask_code-again_btn')}
                        </button>
                    )}
                </div>

                {!processedPhone && (
                    <div className={b('enterPassword')}>
                        <Password
                            label={i18n('ProfileSecurity.security.password_confirm-label')}
                            placeholder={i18n('ProfileSecurity.security.pwd_placeholder')}
                        />
                    </div>
                )}
            </div>
        );
    };

    render() {
        const {
            isMobile,
            phoneToChange,
            hasCodeSent,
            isInProgress,
            isInQuarantine,
            isOldNumberUnAvailable,
            quarantineDaysLeft,
            phonePlaceholder
        } = this.props;

        if (hasCodeSent || (isInProgress && !isInQuarantine)) {
            return this.renderCodeConfirmation();
        }

        if (isInQuarantine) {
            return (
                <div className={b('description')}>
                    <p>{this.getQuarantineMessage(quarantineDaysLeft, phoneToChange)}</p>
                    <p dangerouslySetInnerHTML={{__html: i18n('ProfileSecurity.security.phone_quarantined')}} />
                </div>
            );
        }

        return (
            <form action='#' method='post' onSubmit={this.sendCode}>
                <div className={b('tumbler')}>
                    <Tumbler
                        size='m'
                        checked={isOldNumberUnAvailable}
                        onChange={this.handlePhoneAvailable}
                        labelAfter={i18n('ProfileSecurity.security.number_unavailable')}
                    />
                    <span className={b('tumblerLabelInfo')}>{phoneToChange}</span>
                </div>
                <div className={b('enterNumber')}>
                    <div className={b('input')}>
                        <Field size='l' placeholder={phonePlaceholder} name={FIELDS_NAMES.PHONE} type='tel' />
                    </div>
                    <div className={b('button')}>
                        <Button
                            width={isMobile ? 'max' : 'auto'}
                            size='l'
                            view='action'
                            type='submit'
                            onClick={this.sendCode}
                        >
                            {i18n('ProfileSecurity.security.bind-phone_btn')}
                        </Button>
                    </div>
                </div>
            </form>
        );
    }
}

PhoneReplace.propTypes = {
    enteredPhone: PropTypes.string,
    processedPhone: PropTypes.string,
    availablePhone: PropTypes.string,
    phonePlaceholder: PropTypes.string,
    phoneToChange: PropTypes.string,
    quarantineDaysLeft: PropTypes.number,
    isMobile: PropTypes.bool,
    isInProgress: PropTypes.bool,
    isLoading: PropTypes.bool,
    isOldNumberUnAvailable: PropTypes.bool,
    sendCode: PropTypes.func.isRequired,
    resendCode: PropTypes.func.isRequired,
    getPhoneState: PropTypes.func.isRequired,
    togglePhoneAvailable: PropTypes.func.isRequired,
    hasCodeSent: PropTypes.bool,
    isInQuarantine: PropTypes.bool,
    isReplacedPhoneAvailable: PropTypes.bool
};
