import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';

import {Field} from '@components/Field';
import {Button} from '@components/Button';
import {Spin} from '@components/Spin';

import debounce from 'lodash/debounce';

const b = cn('PublicId');

import './PublicId.styl';
import ReturnToService from '@components/ReturnToService/ReturnToService';

const EXAMPLE_PUBLIC_ID_VALUE = 'yourname';

export class PublicId extends PureComponent {
    componentDidUpdate(prevProps) {
        const {
            hasUpdated,
            retpath,
            isEmbedded,
            isModal,
            savePublicId,
            passportLinks = {},
            firstname,
            lastname,
            suggestPublicId
        } = this.props;

        if (!prevProps.hasUpdated && hasUpdated && !isEmbedded) {
            if (isModal) {
                savePublicId();
                this.handleCancel();
            } else {
                location.href = retpath || passportLinks.profile;
            }
        }

        if (prevProps.firstname !== firstname || prevProps.lastname !== lastname) {
            suggestPublicId();
        }
    }

    componentDidMount() {
        const {suggestPublicId} = this.props;

        suggestPublicId();
    }

    onChangePublicId = () => {
        const {validatePublicId} = this.props;

        validatePublicId();
    };

    onChange = debounce(this.onChangePublicId, 300);

    sendPublicId = (event) => {
        if (event) {
            event.preventDefault();
        }

        this.props.updatePublicId();
    };

    handleCancel = () => {
        const {isModal, closeModal} = this.props;

        if (isModal && typeof closeModal === 'function') {
            closeModal();
        }
    };

    renderTitle = () => {
        const {isEmbedded, isOpened} = this.props;

        if (isEmbedded) {
            return (
                <div className={b('titleWrap', {isEmbedded})} onClick={this.togglePublicId}>
                    <h2 className={b('titleSmall')}>{i18n('Profile.publicId.title')}</h2>
                    <button
                        type='button'
                        onClick={this.togglePublicId}
                        aria-hidden='true'
                        className={b('toggle', {isOpened})}
                    />
                </div>
            );
        }

        return <h1 className={b('title')}>{i18n('Profile.publicId.title')}</h1>;
    };

    togglePublicId = () => {
        const {isOpened, setPublicIdVisible} = this.props;

        setPublicIdVisible(!isOpened);
    };

    onSuggestClick = ({target} = {}) => {
        const {updateValues} = this.props;

        if (target) {
            updateValues({field: 'publicId', value: target.innerText});
            this.onChangePublicId();
        }
    };

    renderInputs = () => {
        const {id, isValid, publicIdSuggest, publicIdSuggestLoading} = this.props;
        const hasSuggestedValues = Boolean(publicIdSuggest && publicIdSuggest.length);
        const isLoading = publicIdSuggestLoading && !hasSuggestedValues;

        return (
            <>
                <div className={b('input')}>
                    <Field
                        shrinkBottomMargin={true}
                        placeholder={id || EXAMPLE_PUBLIC_ID_VALUE}
                        onChange={this.onChange}
                        size='l'
                        maxLength='30'
                        isValid={isValid}
                        validHint={i18n('Profile.publicId.valid')}
                        label={i18n('Profile.publicId.label')}
                        name='publicId'
                    />
                </div>
                <div className={b('inputDescription')}>{i18n('Profile.publicId.inputDescr')}</div>
                <div className={b('suggest', {isLoading})}>
                    {isLoading ? (
                        <Spin progress={true} size='s' />
                    ) : (
                        hasSuggestedValues &&
                        publicIdSuggest.slice(0, 5).map((value) => (
                            <span key={value} className={b('suggestValue')} onClick={this.onSuggestClick}>
                                {value}
                            </span>
                        ))
                    )}
                </div>
                <p className={b('warning')}>{i18n('Profile.publicId.warning')}</p>
            </>
        );
    };

    render() {
        const {
            hasOpenedFromProfile,
            retpath,
            publicIdVal,
            id,
            tld,
            isEmbedded,
            isOpened,
            isModal,
            isNewLayout,
            isLoading
        } = this.props;

        return (
            <div className={b({isEmbedded, page: !isEmbedded && !isModal, nl: isNewLayout, modal: isModal})}>
                {hasOpenedFromProfile && !isModal && (
                    <ReturnToService className={b('returnButton')} retpath={retpath} pathTo='/profile' />
                )}
                {this.renderTitle()}
                <div className={b('wrap', {isHidden: isEmbedded && !isOpened})}>
                    <p className={b('description')}>{i18n('Profile.publicId.text')}</p>
                    <ul className={b('example')}>
                        <li className={b('exampleMarket')}>
                            <span className={'exampleStr'}> {`market.yandex.${tld}/user/`}</span>
                            <span className={b('exampleId')}>{publicIdVal || id || EXAMPLE_PUBLIC_ID_VALUE}</span>
                        </li>
                        <li className={b('exampleQ')}>
                            <span className={'exampleStr'}>{`yandex.${tld}/q/`}</span>
                            <span className={b('exampleId')}>{publicIdVal || id || EXAMPLE_PUBLIC_ID_VALUE}</span>
                        </li>
                    </ul>

                    {isEmbedded ? (
                        <div>{this.renderInputs()}</div>
                    ) : (
                        <form method='post' action='#' onSubmit={this.onSubmit} className={b('form')}>
                            {this.renderInputs()}

                            <div className={b('formButtons')}>
                                {isModal && (
                                    <Button
                                        dataT='cancel-public-id'
                                        size='l'
                                        view='clear'
                                        type='button'
                                        onClick={this.handleCancel}
                                    >
                                        {i18n('_AUTH_.cancel')}
                                    </Button>
                                )}
                                <Button
                                    dataT='save-public-id'
                                    size='l'
                                    view='action'
                                    type='submit'
                                    onClick={this.sendPublicId}
                                    progress={isLoading}
                                >
                                    {i18n('_AUTH_.common.save')}
                                </Button>
                            </div>
                        </form>
                    )}
                </div>
            </div>
        );
    }
}

PublicId.propTypes = {
    publicIdVal: PropTypes.string,
    publicIdSuggest: PropTypes.arrayOf(PropTypes.string),
    tld: PropTypes.string.isRequired,
    passportLinks: PropTypes.object,
    isEmbedded: PropTypes.bool,
    isOpened: PropTypes.bool,
    isModal: PropTypes.bool,
    hasUpdated: PropTypes.bool,
    isLoading: PropTypes.bool,
    isValid: PropTypes.bool,
    publicIdSuggestLoading: PropTypes.bool,
    isNewLayout: PropTypes.bool,
    closeModal: PropTypes.func,
    setPublicIdVisible: PropTypes.func.isRequired,
    savePublicId: PropTypes.func,
    id: PropTypes.string,
    retpath: PropTypes.string,
    firstname: PropTypes.string,
    lastname: PropTypes.string,
    suggestPublicId: PropTypes.func,
    validatePublicId: PropTypes.func.isRequired,
    updatePublicId: PropTypes.func.isRequired,
    updateValues: PropTypes.func,
    hasOpenedFromProfile: PropTypes.bool
};
