import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {Button} from '@components/Button';
import {cn} from '@bem-react/classname';
import {STEPS} from '@blocks/Security/config';
import './FinalScreen.styl';
import metrika from '@blocks/metrics';
import {Link} from '@components/Link';

const b = cn('FinalScreen');

export class FinalScreen extends PureComponent {
    componentDidMount() {
        const metricsDescription = this.getTexts();

        metrika.send(
            `Показ финального экрана, статус "${this.getTitle()}", ${metricsDescription.password.text}, ${
                metricsDescription.phone.text
            }, ${metricsDescription.email.text}`
        );
    }

    getStepToFix = () => {
        const {phone, email, passwordStatus} = this.props;

        if (passwordStatus !== 'ok') {
            return STEPS.PASSWORD;
        }

        if (!phone) {
            return STEPS.PHONE;
        }

        if (!email.length) {
            return STEPS.EMAIL;
        }

        return STEPS.WELCOME;
    };

    goToRetpath = () => {
        const {redirectUrl} = this.props;

        metrika.send('Финальный экран, кнопка завершения (переход на retpath)');
        location.replace(redirectUrl);
    };

    goToFixStep = () => {
        this.props.setNextStep(this.getStepToFix());
        metrika.send('Финальный экран, кнопка "Изменить статус аккаунта" (начать заново)');
    };

    gotoInfoScreen = () => {
        this.props.setNextStep(STEPS.INFO);
    };

    getTitle = () => {
        const {securityLevel} = this.props;

        return securityLevel === 'high'
            ? i18n('ProfileSecurity.security.high_level_title')
            : i18n('ProfileSecurity.security.low_level_title');
    };

    getTexts = () => {
        const {passwordStatus, passwordAge, phone, email} = this.props;
        const texts = {
            password: {
                status: passwordStatus,
                text:
                    passwordStatus === 'ok'
                        ? i18n('ProfileSecurity.password_ok.title')
                        : passwordAge !== 'normal'
                        ? i18n('ProfileSecurity.profile.security.title_pwd-old')
                        : i18n('ProfileSecurity.security.final_pwd_bad'),
                btnText: i18n('_AUTH_.change')
            },
            phone: {
                status: 'warning',
                text: i18n('ProfileSecurity.security.final_phone_not_binded'),
                btnText: i18n('Frontend.common.add')
            },
            email: {
                status: 'warning',
                text: i18n('ProfileSecurity.security.final_email_not_added'),
                btnText: i18n('Frontend.common.add')
            },
            twoFA: {
                status: 'warning',
                text: i18n('ProfileSecurity.security.2fa_not_enabled'),
                btnText: i18n('Frontend.enable')
            }
        };

        if (phone) {
            texts.phone.status = 'ok';
            texts.phone.text = `${i18n('ProfileSecurity.security.final_phone_added')} <span>${phone}</span>`;
            texts.phone.btnText = i18n('_AUTH_.change');
        }

        if (email.length) {
            const emailInfo =
                email.length > 1
                    ? i18n('ProfileSecurity.security.email_info_text')
                          .replace('%1', email[0])
                          .replace('%2', email.length - 1)
                    : email[0];

            texts.email.status = 'ok';
            texts.email.text = `${i18n('ProfileSecurity.security.final_email_added')} <span>${emailInfo}</span>`;
            texts.email.btnText = i18n('_AUTH_.change');
        }

        return texts;
    };

    handleEditClick = (event) => {
        const step = event && event.currentTarget && event.currentTarget.dataset.step;

        if (!step) {
            return;
        }

        this.props.setNextStep(step);
    };

    renderButtons = () => {
        const {securityLevel, isNotMePage, isMobile} = this.props;

        if (securityLevel === 'high') {
            return (
                <Button
                    dataT='goto-profile-btn'
                    size='l'
                    width={isMobile ? 'max' : 'auto'}
                    view='action'
                    type='button'
                    onClick={isNotMePage ? this.gotoInfoScreen : this.goToRetpath}
                >
                    {i18n('ProfileSecurity.final_btn')}
                </Button>
            );
        }

        return (
            <>
                <Button
                    dataT='goto-profile-btn'
                    size='l'
                    width={isMobile ? 'max' : 'auto'}
                    view='clear'
                    type='button'
                    onClick={this.goToRetpath}
                >
                    {i18n('ProfileSecurity.security.skip_settings_btn')}
                </Button>

                <Button
                    dataT='go-to-start-btn'
                    size='l'
                    width={isMobile ? 'max' : 'auto'}
                    view='action'
                    type='button'
                    onClick={this.goToFixStep}
                >
                    {i18n('ProfileSecurity.security.change_settings_btn')}
                </Button>
            </>
        );
    };

    render() {
        const {securityLevel, twoFAUrl} = this.props;
        const descriptionObj = this.getTexts();

        return (
            <main className={b('container')}>
                <div className={b('header')}>
                    <div className={b('header-icon')} role='presentation' aria-hidden='true'>
                        <div className={b('header-avatar')} />
                        <div className={b('shield', {securityLevel})} />
                    </div>
                </div>
                <div className={b('content')}>
                    <h2 className={b('title-large')}>{this.getTitle()}</h2>
                    <div className={b('description')}>
                        <ul className={b('descriptionList')}>
                            {Object.keys(descriptionObj).map((item, i) => {
                                const status = descriptionObj[item].status;
                                const text = descriptionObj[item].text;
                                const btnText = descriptionObj[item].btnText;

                                return (
                                    <li key={i} className={b('descriptionItem', {status, type: item})}>
                                        <div
                                            className={b('descriptionText')}
                                            dangerouslySetInnerHTML={{__html: text}}
                                        />
                                        <div className={b('descriptionBtn')}>
                                            <Link
                                                href={item === 'twoFA' ? twoFAUrl : null}
                                                data-step={item}
                                                onClick={this.handleEditClick}
                                                pseudo={item !== 'twoFA'}
                                                weight='medium'
                                            >
                                                {btnText}
                                            </Link>
                                        </div>
                                    </li>
                                );
                            })}
                        </ul>

                        {securityLevel === 'low' && <p>{i18n('ProfileSecurity.security.improve_settings')}</p>}
                    </div>
                </div>
                <div className={b('footer')}>{this.renderButtons()}</div>
            </main>
        );
    }
}

FinalScreen.propTypes = {
    phone: PropTypes.string,
    email: PropTypes.array,
    passwordStatus: PropTypes.string,
    profileUrl: PropTypes.string,
    twoFAUrl: PropTypes.string,
    passwordAge: PropTypes.string,
    securityLevel: PropTypes.string,
    setNextStep: PropTypes.func.isRequired,
    redirectUrl: PropTypes.string.isRequired,
    isMobile: PropTypes.bool,
    isNotMePage: PropTypes.bool
};
