import {changeSignUpStep, setSuggestToRestoreAccount} from '@blocks/SignUp/actions';
import {updateErrors, updateValues} from '@blocks/actions/form';
import {setProcessError} from '@blocks/UserEntryFlow/actions';
import {getNextStep, getPrevStep, STEP_FIELDS} from './config';
import registrationApi from '@blocks/registration/methods/basicRegistrationMethods';
import api from '@blocks/api';
import {omitEmpty, prettifyPhone, getError, getAdvancedError} from '@blocks/utils';
import {getSuggestedLogins} from '@components/LoginSuggest/actions';
import {requestPhoneConfirmationCode} from '@blocks/actions/phoneConfirm';
import {setAccountsSuggest} from '@blocks/actions/accountsSuggest';
import {YaBroSignUpApi, closeBalloon, generatePassword} from './yabroApi';
import metrika from '@blocks/metrics';
import {domikIsLoading} from '@blocks/auth/actions';

const onNext = (step) => (dispatch, getState) => {
    const fields = STEP_FIELDS[step] || [];
    const state = getState();
    const {form = {}, tracks = {}, common = {}} = state;
    const {values = {}, errors = {}, suggestedPassword, type: formType} = form;
    const {registerTrackId: trackId} = tracks;
    const {phone, phoneCode, firstname, lastname} = values;
    const {csrf, origin} = common;
    const nextStep = getNextStep(step);
    const formErrors = [];

    metrika.send(`Клик "Далее" на экране ${step}`);

    fields.forEach((field) => {
        if (!values[field] || (errors[field] && errors[field].code)) {
            formErrors.push({
                field,
                error: errors[field] && errors[field].code ? errors[field] : getError(field, 'missingvalue')
            });
            metrika.send(`Показ ошибки поля ${field}: missingvalue`);
        }
    });

    if (formErrors.length) {
        formErrors.forEach((error) => dispatch(updateErrors(error)));
        return;
    }

    if (step === 'personalData') {
        api.validatePhone({trackId, phone}).then((response) =>
            requestPhoneConfirmationCode({dispatch, getState, phoneValidateParams: response}).then(() => {
                dispatch(changeSignUpStep(nextStep));
            })
        );
    }

    if (step === 'phoneConfirm') {
        api.checkPhoneConfirmationCode({trackId, code: phoneCode})
            .then(() => {
                api.findAccountsByNameAndPhone({trackId, firstname, lastname})
                    .then((response = {}) => {
                        const {accounts = []} = response;

                        if (accounts.length) {
                            dispatch(setSuggestToRestoreAccount(true));
                            dispatch(setAccountsSuggest(accounts));
                            return;
                        }

                        getSuggestedLogins(dispatch, state, true).then(() => dispatch(changeSignUpStep(nextStep)));
                    })
                    .catch(() =>
                        getSuggestedLogins(dispatch, state, true).then(() => dispatch(changeSignUpStep(nextStep)))
                    );
            })
            .catch((error = {}) => {
                const code = error.errors && error.errors[0];

                dispatch(updateErrors({field: 'phoneCode', error: getError('phoneCode', code)}));
                metrika.send(`Показ ошибки поля phoneCode: ${code}`);
            });
    }

    if (step === 'login') {
        generatePassword()(dispatch);
        dispatch(changeSignUpStep(nextStep));
    }

    if (step === 'eula') {
        const data = Object.assign({}, values, {
            'human-confirmation': 'phone',
            trackId,
            csrf_token: csrf,
            eula_accepted: values.eula_accepted ? 'y' : 'n'
        });

        if (formType) {
            data.type = formType;
        }

        if (origin) {
            data.origin = origin;
        }

        registrationApi
            .asyncRegistration(omitEmpty(data), dispatch)
            .then((response = {}) => {
                const {status} = response;

                if (status !== 'ok') {
                    dispatch(setProcessError(getAdvancedError('registration', 'smthWrong')));
                    dispatch(domikIsLoading(false));
                    return;
                }

                if (suggestedPassword === data.password) {
                    YaBroSignUpApi.saveSignupDataForLogin(data.login);
                }

                dispatch(changeSignUpStep(nextStep));
            })
            .catch((responseError = {}) => {
                const errors = responseError.errors || responseError.error || [];
                const code = typeof errors[0] === 'string' ? errors[0] : (errors[0] || {}).code;

                dispatch(domikIsLoading(false));
                metrika.send(`Ошибка регистрации ${code}`);

                if (code) {
                    dispatch(setProcessError(getAdvancedError('registration', code)));
                }
            });
    }

    if (step === 'finish') {
        closeBalloon();
    }
};

const onBack = (step) => (dispatch) => {
    dispatch(changeSignUpStep(getPrevStep(step)));
    metrika.send(`Клик "Назад" на экране ${step}`);
};

const getPersonalData = () => (dispatch) => {
    YaBroSignUpApi.getPersonalData((data = {}) => {
        const {firstName, lastName, phone} = data;

        if (firstName) {
            dispatch(updateValues({field: 'firstname', value: firstName}));
        }

        if (lastName) {
            dispatch(updateValues({field: 'lastname', value: lastName}));
        }

        if (phone) {
            dispatch(updateValues({field: 'phone', value: prettifyPhone(phone)}));
        }
    });
};

export {onNext, onBack, getPersonalData};
