import './UserEntryFlow.styl';
import {getAdvancedError} from '@blocks/utils';
import React from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import SocialButtons from '@blocks/registration/socialButtons/socialButtons';
import {Button} from '@components/Button';
import {Checkbox} from '@components/Checkbox';
import {Popup} from '@components/Popup';
import {Icon} from '@components/Icon';
import {GlobalError} from '@screens/GlobalError';
import {Captcha} from '@screens/Captcha';
import api from '@blocks/api';
import {isRestoreProcess, getSteps, isValidStep, isRegisterProcess} from './utils';
import {STEP_SCREENS} from './screens';
import {
    COMMON_STEPS,
    REGISTER_NEOPHONISH_STEPS,
    REGISTER_STEPS,
    REGISTER_WITH_FULL_PERSONAL_INFO_STEPS,
    REGISTER_NEOPHONISH_WITH_FULL_PERSONAL_INFO_STEPS,
    STEPS
} from './steps';
import {
    ENTRY_RESTORE_PROCESS,
    ENTRY_REGISTER_PROCESS,
    ENTRY_REGISTER_PORTAL_PROCESS,
    ENTRY_REGISTER_NO_PHONE_PROCESS,
    ENTRY_REGISTER_NEOPHONISH_PROCESS,
    ENTRY_RESTORE_NEOPHONISH_PROCESS,
    ENTRY_REGISTER_WITH_FULL_PERSONAL_INFO_PROCESS,
    ENTRY_REGISTER_WITH_FULL_PERSONAL_INFO_NEOPHONISH_PROCESS,
    SCENARIO_TYPE_REGISTRATION,
    SCENARIO_TYPE_AUTH
} from './processes';
import {DomikFooterLink} from '@blocks/authv2/components/DomikFooterLink';

const b = cn('UserEntryFlow');

class UserEntryFlow extends React.PureComponent {
    state = {showLegalTooltip: false};
    checkboxRef = React.createRef();
    componentDidMount() {
        const {
            trackId,
            setProcessError,
            setMetricsHeader,
            process,
            step,
            restartProcess,
            canSendFingerprint,
            sendFingerprint
        } = this.props;

        setMetricsHeader();

        if (!isValidStep(process, step)) {
            restartProcess();
            return;
        }

        if (trackId) {
            this.writeStatbox(trackId);
            return;
        }

        api.userEntryFlowSubmit({process})
            .then((response = {}) => {
                this.writeStatbox(response.id);

                if (canSendFingerprint) {
                    sendFingerprint();
                }
            })
            .catch(() => {
                setProcessError(getAdvancedError('global', 'global'));
            });
    }

    componentWillUnmount() {
        const {safeStateClearing} = this.props;

        safeStateClearing();
    }

    writeStatbox = (trackId) => {
        api.writeStatbox({
            track_id: trackId,
            mode: this.getMode(),
            action: 'opened'
        });
    };

    getMode = () => {
        const {process} = this.props;

        switch (process) {
            case ENTRY_RESTORE_PROCESS: {
                return '2step_domik_restore_login';
            }

            case ENTRY_REGISTER_PROCESS: {
                return 'registration_2step_domik';
            }

            case ENTRY_REGISTER_WITH_FULL_PERSONAL_INFO_PROCESS: {
                return 'registration_2step_domik_with_full_personal_info';
            }

            case ENTRY_REGISTER_NEOPHONISH_PROCESS: {
                return 'registration_neophonish_2step_domik';
            }

            case ENTRY_REGISTER_WITH_FULL_PERSONAL_INFO_NEOPHONISH_PROCESS: {
                return 'registration_neophonish_2step_domik_with_full_personal_info';
            }

            case ENTRY_RESTORE_NEOPHONISH_PROCESS: {
                return 'restore_neophonish_2step_domik';
            }

            default: {
                return '2step_domik';
            }
        }
    };

    onNext = (event) => {
        const {
            step,
            process,
            phone,
            onNext,
            useNewSuggestByPhone,
            isSkipEulaPage,
            isEulaAccepted,
            processError,
            checkFioStepIsRequired
        } = this.props;

        if (event) {
            event.preventDefault();
        }

        if (
            isSkipEulaPage &&
            step === STEPS.PHONE &&
            process === ENTRY_REGISTER_NEOPHONISH_PROCESS &&
            processError.status === 'ok' &&
            !isEulaAccepted
        ) {
            return this.setState({showLegalTooltip: true});
        }

        if (step === STEPS.PHONE && useNewSuggestByPhone && phone) {
            checkFioStepIsRequired(phone, isRestoreProcess(process) ? SCENARIO_TYPE_AUTH : SCENARIO_TYPE_REGISTRATION);
        }

        onNext(step);
    };

    onAccountsNext = (event) => {
        const {process, switchToModeRegister} = this.props;

        if (isRestoreProcess(process)) {
            switchToModeRegister();
        }

        this.onNext(event);
    };

    getStepBody = () => {
        const {processError = {}, isCaptchaRequired} = this.props;

        if (processError.status && processError.status !== 'ok') {
            return this.renderError();
        }

        if (isCaptchaRequired) {
            return this.renderCaptcha();
        }

        return this.renderFormStep();
    };

    getScreen = () => {
        const {
            step,
            signUpMode,
            retryToRequestPhoneConfirmationCode,
            trackId,
            onBack,
            hasChildishExp,
            isAm
        } = this.props;
        const Screen = STEP_SCREENS[step];
        const props = {
            place: signUpMode
        };

        if (step === STEPS.PHONE_CONFIRM) {
            props.onRetry = retryToRequestPhoneConfirmationCode;
            props.goBack = onBack;
        }

        if (step === STEPS.ACCOUNTS) {
            props.trackId = trackId;
            props.onContinue = this.onAccountsNext;
        }

        if (step === STEPS.PHONE) {
            props.showChildishLanding = hasChildishExp && !isAm;
        }

        return <Screen onConfirm={this.onNext} {...props} />;
    };

    maybeRenderSocialButtons = () => {
        const {signUpMode, step, process, useNewSuggestByPhone} = this.props;
        const steps = getSteps(process, useNewSuggestByPhone);

        if ((!step || steps[0] === step) && isRegisterProcess(process)) {
            return <SocialButtons place={signUpMode} />;
        }
    };

    renderFormStep = () => {
        const {trackId} = this.props;

        return (
            <React.Fragment>
                <form autoComplete='off' method='post' onSubmit={this.onNext} className={b('form')}>
                    <input name='track_id' id='track_id' type='hidden' value={trackId} />
                    {this.getScreen()}
                </form>
                {this.maybeRenderSocialButtons()}
            </React.Fragment>
        );
    };

    renderError = () => {
        const {restartProcess, processError, signUpMode} = this.props;

        return <GlobalError onRestart={restartProcess} error={processError} place={signUpMode} />;
    };

    renderCaptcha = () => {
        return <Captcha onConfirmCallback={this.onNext} />;
    };

    switchToLoginPage = (event) => {
        const {
            updateLoginValue,
            changePagePopupVisibility,
            changePagePopupType,
            changeNativeInputValue,
            switchToModeAddingAccount
        } = this.props;

        event.preventDefault();
        updateLoginValue('');
        changePagePopupVisibility(false);
        changePagePopupType('');
        changeNativeInputValue('login', '');
        switchToModeAddingAccount();
    };

    isFirstStep = () => {
        const {process, step, useNewSuggestByPhone} = this.props;
        const steps = getSteps(process, useNewSuggestByPhone);

        return !step || step === steps[0];
    };

    getFooterText = () => {
        const {process} = this.props;
        const isRestore = process === ENTRY_RESTORE_PROCESS;
        const isRegisterFirstStep = isRegisterProcess(process) && this.isFirstStep();

        if (isRegisterFirstStep) {
            return i18n('_AUTH_.preauth.sign_in.v2');
        }

        if (isRestore) {
            return i18n('_AUTH_.remember_id');
        }
    };
    onLegalChange = () => {
        const {updateValues, isEulaAccepted} = this.props;

        this.setState({showLegalTooltip: false});
        updateValues({field: 'eula_accepted', value: !isEulaAccepted});
    };
    maybeRenderLegal = () => {
        const {
            isSkipEulaPage,
            step,
            process,
            processError = {},
            isEulaAccepted,
            eulaTexts: {description = ''} = {}
        } = this.props;
        const {showLegalTooltip} = this.state;

        if (
            isSkipEulaPage &&
            step === STEPS.PHONE &&
            process === ENTRY_REGISTER_NEOPHONISH_PROCESS &&
            processError.status === 'ok'
        ) {
            return (
                <>
                    <div className={b('eula')}>
                        <label className={b('label')}>
                            <Checkbox
                                innerRef={this.checkboxRef}
                                className={b('checkbox')}
                                id='eula_legal'
                                name='eula_legal'
                                checked={isEulaAccepted}
                                onChange={this.onLegalChange}
                                size='m'
                            />
                            <span className={b('description')} dangerouslySetInnerHTML={{__html: description}} />
                        </label>
                    </div>
                    <Popup
                        visible={showLegalTooltip}
                        directions={[
                            'top-left',
                            'top-center',
                            'top-right',
                            'bottom-left',
                            'bottom-center',
                            'bottom-right',
                            'top-left'
                        ]}
                        hasTail={true}
                        target='anchor'
                        anchor={this.checkboxRef}
                        zIndex={9999}
                        view='black'
                    >
                        <div className={b('popup')}>{i18n('_AUTH_.registration.step.phone.gdpr_popup')}</div>
                    </Popup>
                </>
            );
        }
    };

    maybeRenderFooterLink = () => {
        const {process, addUserFullUrl, isAmModePhonish, isSkipEulaPage, hideNeophonishOtherAccountLink} = this.props;
        const isRestore = process === ENTRY_RESTORE_PROCESS;
        const isRegisterFirstStep = isRegisterProcess(process) && this.isFirstStep();
        const showLink = isRestore || isRegisterFirstStep;

        if (hideNeophonishOtherAccountLink || !showLink) {
            return null;
        }

        const showButton = isAmModePhonish && !isSkipEulaPage;

        return showButton ? (
            <Button
                type='button'
                view='pseudo'
                size='l'
                width='max'
                className='UserEntryFlow__sign-in-with-yandex-btn'
                onClick={this.switchToLoginPage}
            >
                <Icon type='yandex' size='s' place='button' />
                {i18n('_AUTH_.social_sign_in').replace('%1', i18n('_AUTH_.common.yandex'))}
            </Button>
        ) : (
            <DomikFooterLink href={addUserFullUrl} onClick={this.switchToLoginPage} pseudo={true}>
                {this.getFooterText()}
            </DomikFooterLink>
        );
    };

    render() {
        return (
            <div className={b()} id={b()}>
                {this.getStepBody()}
                {this.maybeRenderLegal()}
                {this.maybeRenderFooterLink()}
            </div>
        );
    }
}

UserEntryFlow.propTypes = {
    trackId: PropTypes.string,
    phone: PropTypes.string,
    process: PropTypes.oneOf([
        ENTRY_RESTORE_PROCESS,
        ENTRY_REGISTER_PROCESS,
        ENTRY_REGISTER_PORTAL_PROCESS,
        ENTRY_REGISTER_NO_PHONE_PROCESS,
        ENTRY_REGISTER_NEOPHONISH_PROCESS,
        ENTRY_RESTORE_NEOPHONISH_PROCESS,
        ENTRY_REGISTER_WITH_FULL_PERSONAL_INFO_PROCESS,
        ENTRY_REGISTER_WITH_FULL_PERSONAL_INFO_NEOPHONISH_PROCESS
    ]),
    step: PropTypes.oneOf(
        [].concat(
            COMMON_STEPS,
            REGISTER_STEPS,
            REGISTER_NEOPHONISH_STEPS,
            REGISTER_WITH_FULL_PERSONAL_INFO_STEPS,
            REGISTER_NEOPHONISH_WITH_FULL_PERSONAL_INFO_STEPS,
            STEPS.SQ_SA
        )
    ).isRequired,
    onNext: PropTypes.func.isRequired,
    onBack: PropTypes.func.isRequired,
    retryToRequestPhoneConfirmationCode: PropTypes.func.isRequired,
    restartProcess: PropTypes.func.isRequired,
    processError: PropTypes.shape({
        status: PropTypes.string,
        text: PropTypes.string
    }),
    eulaTexts: PropTypes.object,
    signUpMode: PropTypes.string,
    addUserFullUrl: PropTypes.string,
    switchToModeRegister: PropTypes.func.isRequired,
    setProcessError: PropTypes.func.isRequired,
    setMetricsHeader: PropTypes.func.isRequired,
    safeStateClearing: PropTypes.func.isRequired,
    updateLoginValue: PropTypes.func.isRequired,
    updateValues: PropTypes.func.isRequired,
    useNewSuggestByPhone: PropTypes.bool,
    changePagePopupVisibility: PropTypes.func.isRequired,
    changePagePopupType: PropTypes.func.isRequired,
    changeNativeInputValue: PropTypes.func.isRequired,
    switchToModeAddingAccount: PropTypes.func.isRequired,
    isCaptchaRequired: PropTypes.bool.isRequired,
    isEulaAccepted: PropTypes.bool,
    isAmModePhonish: PropTypes.bool,
    isAm: PropTypes.bool,
    hideNeophonishOtherAccountLink: PropTypes.bool,
    isSkipEulaPage: PropTypes.bool,
    sendFingerprint: PropTypes.func.isRequired,
    checkFioStepIsRequired: PropTypes.func.isRequired,
    canSendFingerprint: PropTypes.bool,
    hasChildishExp: PropTypes.bool
};

export {UserEntryFlow};
