import {push} from 'connected-react-router';
import {getFormErrors, getFormValues, getRedirectUrl, getTrackId, hasExp, isAuthComplete} from '@blocks/selectors';
import {domikIsLoading} from '@blocks/auth/actions';
import {updateErrors, updateValues, updateStates, setCaptchaRequired} from '@blocks/actions/form';
import {toggleBackPaneVisibility} from '@blocks/common/actions';
import {requestPhoneConfirmationCode} from '@blocks/actions/phoneConfirm';
import {setAccountsSuggest} from '@blocks/actions/accountsSuggest';
import {TRACK_TYPES} from '@blocks/actions/tracks';
import {initAdditionalDataRequest, setupBackPane} from '@blocks/authv2/actions';
import {getSuggestedLogins} from '@components/LoginSuggest/actions';
import {getError, getAdvancedError, omitEmpty} from '@blocks/utils';
import registrationApi from '@blocks/registration/methods/basicRegistrationMethods';
import {ACCOUNT_TYPES, getAccountTypeByAlias} from '@blocks/authv2/utils/accountTypes';
import {neoPhonishAuth} from '@blocks/actions/neoPhonish';
import {FIELDS_NAMES} from '@components/Field/names';
import metrika from '@blocks/metrics';
import api from '@blocks/api';
import {shouldForceConfirmPhoneBySMS, amFlashcallExpOn} from '@blocks/authv2/nativeMobileApi/helpers';
import {amSetAnalyticsToRetpath} from '@blocks/authv2/actions/nativeMobileApi';
import {validateSecurityAnswer} from './actions/validateSecurityAnswer';
import {
    changeStep as originalChangeStep,
    setProcessError,
    clearProcessError,
    setFioStepRequired,
    setAllowedRegFlow,
    changeProcess,
    setIsRegAfterAuthStart
} from './actions';
import {STEPS, getFields} from './steps';
import {
    getProcessName,
    isRegisterProcess,
    isRestoreProcess,
    getNextStep,
    getPrevStep,
    getSteps,
    isRegisterNeophonishProcess,
    checkIsDoregishFlowProcess,
    checkIsNewSuggestByPhone
} from './utils';
import {
    ENTRY_REGISTER_NEOPHONISH_PROCESS,
    ENTRY_RESTORE_NEOPHONISH_PROCESS,
    ENTRY_RESTORE_PROCESS,
    ENTRY_REGISTER_WITH_FULL_PERSONAL_INFO_NEOPHONISH_PROCESS,
    ENTRY_REGISTER_WITH_FULL_PERSONAL_INFO_PROCESS,
    ENTRY_REGISTER_PROCESS,
    ENTRY_REGISTER_PORTAL_PROCESS,
    SCENARIO_TYPE_REGISTRATION
} from './processes';

const SMS_ONLY_PROCESS = [
    ENTRY_REGISTER_NEOPHONISH_PROCESS,
    ENTRY_RESTORE_NEOPHONISH_PROCESS,
    ENTRY_REGISTER_WITH_FULL_PERSONAL_INFO_NEOPHONISH_PROCESS
];

const changeStepThunk = (nextStep) => (dispatch, getState) => {
    const state = getState();
    const {am = {}} = state;

    dispatch(originalChangeStep(nextStep));

    if (!am.isAm) {
        // не показываем кнопку назад на шаге с найденными аккаунтами
        dispatch(toggleBackPaneVisibility(nextStep !== STEPS.ACCOUNTS));
    }
};

const changeStep = (dispatch, nextStep) => {
    dispatch(changeStepThunk(nextStep));
};

export const initWithPhoneConfirmation = ({phone, route, processName, isRegAfterAuthStart = false}) => (dispatch) => {
    const callbackAction = () => {
        changeStep(dispatch, STEPS.PHONE_CONFIRM);
        dispatch(setIsRegAfterAuthStart(isRegAfterAuthStart));
        dispatch(push(route));
    };

    dispatch(updateValues({field: FIELDS_NAMES.PHONE, value: phone}));
    dispatch(confirmPhone(callbackAction, processName, true));
};

export const confirmPhone = (callback, processName) => (dispatch, getState) => {
    const state = getState();
    const values = getFormValues(state);
    const {am: {isAm} = {}} = state;
    const useNewSuggestByPhone = checkIsNewSuggestByPhone(state);
    const process = getProcessName(state);
    const trackId = getTrackId(
        state,
        (useNewSuggestByPhone && !process && processName === ENTRY_REGISTER_PROCESS) || isAuthComplete(state)
            ? TRACK_TYPES.COMMON
            : TRACK_TYPES.REGISTER
    );
    const {phone} = values;
    const forceSMS =
        useNewSuggestByPhone ||
        SMS_ONLY_PROCESS.includes(processName) ||
        SMS_ONLY_PROCESS.includes(process) ||
        shouldForceConfirmPhoneBySMS(state);

    dispatch(domikIsLoading(true));

    return api
        .validatePhone({trackId, phone})
        .then((response = {}) => {
            const params = {dispatch, getState, trackId};
            const {status, errors = [], isPhoneValidForCall, isPhoneValidForFlashCall} = response;

            if (status === 'error') {
                const code = errors[0];

                dispatch(
                    updateErrors({
                        field: FIELDS_NAMES.PHONE,
                        error: getError(FIELDS_NAMES.PHONE, code)
                    })
                );
                dispatch(domikIsLoading(false));
                return;
            }

            if (!forceSMS) {
                params.phoneValidateParams = {};

                if (isPhoneValidForFlashCall) {
                    params.phoneValidateParams.isPhoneValidForFlashCall =
                        (process === ENTRY_RESTORE_PROCESS && hasExp(state, 'restorelogin_call_flashcall')) ||
                        (isRegisterProcess(process) && hasExp(state, 'passp-20231-hangup-touch')) ||
                        amFlashcallExpOn(state);
                }

                if (isPhoneValidForCall && !isAm) {
                    params.phoneValidateParams.isPhoneValidForCall =
                        (process === ENTRY_RESTORE_PROCESS && hasExp(state, 'restorelogin_call_dictcall')) ||
                        (isRegisterProcess(process) && hasExp(state, 'passp-20231-forced-zvonok'));
                }
            }

            return requestPhoneConfirmationCode(params)
                .then(() => {
                    if (typeof callback === 'function') {
                        callback();
                    }

                    dispatch(domikIsLoading(false));
                })
                .catch((error = {}) => {
                    const code = error.errors && error.errors[0];

                    if (code === 'captcha.not_checked' || code === 'captcha.required') {
                        dispatch(setCaptchaRequired(true));
                    }

                    dispatch(domikIsLoading(false));
                });
        })
        .catch(() => {
            dispatch(domikIsLoading(false));
        });
};

export const checkFioStepIsRequired = (phone, scenario = SCENARIO_TYPE_REGISTRATION) => (dispatch, getState) => {
    const state = getState();
    const {common = {}, tracks = {}} = state;
    const trackId = common.track_id || tracks.registerTrackId;

    return api.checkPhoneBySquatter({csrf: common.csrf, trackId, phone, scenario}).then((response = {}) => {
        if (response.status === 'ok') {
            metrika.send(`Проверка номера на мониторинг, требуется ввод ФИ: ${response.require_flow_with_fio}`);
            return dispatch(setFioStepRequired(response.require_flow_with_fio));
        }

        metrika.send(`Проверка номера на мониторинг, ошибка ${response}`);
    });
};

const registerNeoPhonish = ({data, dispatch, options, isAm}) => {
    const {redirectUrl} = options;

    registrationApi
        .registerNeoPhonish(omitEmpty(data))
        .then((response = {}) => {
            const {status} = response;

            if (isAm) {
                dispatch(domikIsLoading(false));
            }

            if (status !== 'ok') {
                dispatch(setProcessError(getAdvancedError('registration', 'smthWrong')));
                dispatch(domikIsLoading(false));
                return;
            }

            return location.replace(redirectUrl);
        })
        .catch((registrationError = {}) => {
            const errors = registrationError.errors || registrationError.error || [];
            const code = typeof errors[0] === 'string' ? errors[0] : (errors[0] || {}).code;

            dispatch(setProcessError(getAdvancedError('registration', code)));
            dispatch(domikIsLoading(false));
        });
};

const registerAccount = ({data, dispatch, options, isAm}) => {
    const {redirectUrl, isAvatarRequestExp, askAvatarUrl} = options;
    const {trackId} = data;

    registrationApi
        .asyncRegistration(omitEmpty(data), dispatch)
        .then((response = {}) => {
            const {status} = response;

            if (isAm) {
                dispatch(domikIsLoading(false));
            }

            if (status !== 'ok') {
                dispatch(setProcessError(getAdvancedError('registration', 'smthWrong')));
                dispatch(domikIsLoading(false));
                return;
            }

            if (!isAvatarRequestExp || isAm) {
                return location.replace(redirectUrl);
            }

            return api
                .updateCsrfToken()
                .then(() => {
                    dispatch(
                        initAdditionalDataRequest({
                            action: 'add',
                            state: 'avatar',
                            track_id: trackId
                        })
                    );
                    dispatch(setupBackPane(null));
                    dispatch(domikIsLoading(false));
                    dispatch(push(askAvatarUrl));
                })
                .catch(() => location.replace(redirectUrl));
        })
        .catch((responseError = {}) => {
            const errors = responseError.errors || responseError.error || [];
            const code = typeof errors[0] === 'string' ? errors[0] : (errors[0] || {}).code;

            dispatch(domikIsLoading(false));
            metrika.send(`Ошибка регистрации ${code}`);

            if (code) {
                dispatch(setProcessError(getAdvancedError('registration', code)));
            }
        });
};

export const onNext = (step) => async (dispatch, getState) => {
    const state = getState();
    const values = getFormValues(state);
    const errors = getFormErrors(state);
    const process = getProcessName(state);
    const fields = getFields(step, state);
    const trackId = getTrackId(state, TRACK_TYPES.REGISTER);
    const isAvatarRequestExp = hasExp(state, 'avatar-dozapros-newreg-exp1');
    const {
        common = {},
        form = {},
        am = {},
        customs = {},
        userEntryFlow: {allowedRegFlow, isRegAfterAuthStart, isFioStepRequired} = {}
    } = state;
    const {askAvatarUrl, csrf, origin} = common;
    const {phoneCode, firstname, lastname} = values;
    const {type: formType, keepUnsubscribedValue, validation, states} = form;
    const formErrors = [];
    const {isAm} = am;
    const isDoregishFlowAvailable = checkIsDoregishFlowProcess(state);
    const nextStep = getNextStep(step, process, isDoregishFlowAvailable, allowedRegFlow);

    let redirectUrl = getRedirectUrl(state);

    const switchToPortalReg = () => (dispatch) => {
        dispatch(setFioStepRequired(true));
        dispatch(changeProcess(ENTRY_REGISTER_PORTAL_PROCESS));
        changeStep(dispatch, STEPS.PERSONAL_DATA);
        dispatch(domikIsLoading(false));
    };
    const regAccount = () => {
        const data = Object.assign({}, values, {
            'human-confirmation': validation.method,
            trackId,
            csrf_token: csrf,
            eula_accepted: values.eula_accepted ? 'y' : 'n'
        });

        if (formType && isRegisterProcess(process) && !isRegisterNeophonishProcess(process)) {
            data.type = formType;
        }

        if (keepUnsubscribedValue) {
            data.unsubscribe_from_maillists = keepUnsubscribedValue;
        }

        if (origin) {
            data.origin = origin;
        }

        if (isRegisterNeophonishProcess(process)) {
            if (!isFioStepRequired) {
                /* такая ситуация, что мы могли раньше выпросить у пользователя ФИ,
                и тут вдруг сквоттер говорит "а давайте дорегиша сделаем!" тут-то мы и удалим ФИ */
                delete data.firstname;
                delete data.lastname;
            }

            if (isAm) {
                dispatch(amSetAnalyticsToRetpath('reg_neo_phonish'));

                redirectUrl = getRedirectUrl(getState());
            }

            registerNeoPhonish({data, dispatch, isAm, options: {redirectUrl}});
            return;
        }

        if (isAm) {
            dispatch(amSetAnalyticsToRetpath('Registration'));

            redirectUrl = getRedirectUrl(getState());
        }

        registerAccount({data, dispatch, isAm, options: {redirectUrl, isAvatarRequestExp, askAvatarUrl}});
    };

    const findAccountsAndGoNext = () =>
        (isDoregishFlowAvailable
            ? api.findAccountsByPhone({trackId})
            : api.findAccountsByNameAndPhone({trackId, firstname, lastname, allowNeophonish: true})
        )
            .then((response = {}) => {
                const {accounts = [], allowed_registration_flows = []} = response;

                if (!allowedRegFlow) {
                    metrika.send(`Доступный вид регистрации: '${allowed_registration_flows[0]}'`);
                    dispatch(setAllowedRegFlow(allowed_registration_flows[0] || ''));
                }

                dispatch(setAccountsSuggest(accounts));

                if (!isRegisterNeophonishProcess(process) && allowed_registration_flows.includes('neophonish')) {
                    const isEulaStepNext = !accounts.length && !isRegAfterAuthStart && !isRestoreProcess(process);

                    dispatch(changeProcess(ENTRY_REGISTER_NEOPHONISH_PROCESS));
                    changeStep(dispatch, isEulaStepNext ? STEPS.EULA : nextStep);
                    dispatch(domikIsLoading(false));
                    return;
                }

                if (!isRegisterNeophonishProcess(process) && allowed_registration_flows[0] === 'portal') {
                    dispatch(changeProcess(ENTRY_REGISTER_PORTAL_PROCESS));
                    changeStep(dispatch, nextStep);
                    dispatch(domikIsLoading(false));
                    return;
                }

                // на регистрации скипаем пустой экран саджеста аккаунтов,
                // если это не регистрация неофониша после авторизации
                if (!accounts.length && isRegisterProcess(process)) {
                    if (customs.isSkipEulaPage && nextStep === STEPS.EULA) {
                        regAccount();
                        return;
                    }

                    if (
                        (isRegisterNeophonishProcess(process) || allowed_registration_flows.includes('neophonish')) &&
                        isRegAfterAuthStart
                    ) {
                        dispatch(domikIsLoading(false));
                        changeStep(dispatch, nextStep);
                        return;
                    }

                    return getSuggestedLogins(dispatch, state, true)
                        .then(() => {
                            const realNextStep = getNextStep(
                                nextStep,
                                process,
                                isDoregishFlowAvailable,
                                allowedRegFlow
                            );

                            if (customs.isSkipEulaPage && realNextStep === STEPS.EULA) {
                                regAccount();
                                return;
                            }

                            changeStep(dispatch, process === ENTRY_REGISTER_PORTAL_PROCESS ? nextStep : realNextStep);
                            dispatch(domikIsLoading(false));
                        })
                        .catch(() => dispatch(domikIsLoading(false)));
                }

                // автологиним в неофониш если на их регистрации или восстановлении и в саджесте только неофониш
                if (
                    accounts.length === 1 &&
                    (isRegisterNeophonishProcess(process) || process === ENTRY_RESTORE_NEOPHONISH_PROCESS)
                ) {
                    const account = accounts[0];
                    const {
                        primary_alias_type: primaryAliasType,
                        uid,
                        allowed_auth_flows: allowedAuthFlows = []
                    } = account;
                    const isInstantAuthAvailable = allowedAuthFlows.includes('instant');

                    if (
                        getAccountTypeByAlias(primaryAliasType) === ACCOUNT_TYPES.NEOPHONISH &&
                        isInstantAuthAvailable
                    ) {
                        metrika.send('Инстант-вход пользователя через саджест аккаунтов');
                        dispatch(neoPhonishAuth({trackId, uid, lastname, firstname}));
                        return;
                    }
                }

                changeStep(dispatch, nextStep);
                dispatch(domikIsLoading(false));
            })
            .catch((errorResponse = {}) => {
                const {errors = []} = errorResponse;
                const error = errors[0];

                if (isRegisterProcess(process) && !isRegisterNeophonishProcess(process)) {
                    return getSuggestedLogins(dispatch, state, true)
                        .then(() => {
                            changeStep(
                                dispatch,
                                getNextStep(nextStep, process, isDoregishFlowAvailable, allowedRegFlow)
                            );
                            dispatch(domikIsLoading(false));
                        })
                        .catch(() => dispatch(domikIsLoading(false)));
                }

                if ((isRestoreProcess(process) || isRegisterNeophonishProcess(process)) && error) {
                    dispatch(setProcessError(getAdvancedError('findAccountsByNameAndPhone', error)));
                    dispatch(domikIsLoading(false));
                    return;
                }

                dispatch(domikIsLoading(false));
            });

    dispatch(setCaptchaRequired(false));
    dispatch(domikIsLoading(true));

    metrika.send(`Клик "Далее" на экране ${step}`);

    fields.forEach((field) => {
        if (
            !values[field] ||
            (values[field] && !values[field].trim()) ||
            (errors[field] && errors[field].code && errors[field].code !== 'captcha.required')
        ) {
            formErrors.push({
                field,
                error: errors[field] && errors[field].code ? errors[field] : getError(field, 'missingvalue')
            });
            metrika.send(`Показ ошибки поля ${field}: missingvalue`);
        }
    });

    if (formErrors.length) {
        formErrors.forEach((error) => dispatch(updateErrors(error)));
        dispatch(domikIsLoading(false));
        return;
    }

    if (step === STEPS.PHONE || step === STEPS.PERSONAL_DATA_AND_PHONE) {
        dispatch(confirmPhone(() => changeStep(dispatch, nextStep)));
    }

    if (step === STEPS.PHONE_CONFIRM) {
        await api
            .checkPhoneConfirmationCode({trackId, code: phoneCode})
            .then(() => {
                if (
                    [
                        ENTRY_REGISTER_WITH_FULL_PERSONAL_INFO_NEOPHONISH_PROCESS,
                        ENTRY_REGISTER_WITH_FULL_PERSONAL_INFO_PROCESS
                    ].includes(process) ||
                    (isDoregishFlowAvailable &&
                        [
                            ENTRY_REGISTER_PROCESS,
                            ENTRY_REGISTER_NEOPHONISH_PROCESS,
                            ENTRY_RESTORE_NEOPHONISH_PROCESS,
                            ENTRY_RESTORE_PROCESS
                        ].includes(process))
                ) {
                    return findAccountsAndGoNext();
                }

                changeStep(dispatch, nextStep);
                dispatch(domikIsLoading(false));
            })
            .catch((error = {}) => {
                const code = error.errors && error.errors[0];

                dispatch(
                    updateErrors({field: FIELDS_NAMES.PHONE_CODE, error: getError(FIELDS_NAMES.PHONE_CODE, code)})
                );
                dispatch(domikIsLoading(false));
                metrika.send(`Показ ошибки поля phoneCode: ${code}`);
            });
    }

    if (step === STEPS.PERSONAL_DATA) {
        if (allowedRegFlow === 'portal') {
            dispatch(domikIsLoading(false));
            return changeStep(dispatch, nextStep);
        }

        await findAccountsAndGoNext();
    }

    if (step === STEPS.ACCOUNTS) {
        if (allowedRegFlow === 'portal' && isDoregishFlowAvailable) {
            dispatch(changeProcess(ENTRY_REGISTER_PROCESS));
            dispatch(switchToPortalReg(dispatch));
            return;
        }

        if (nextStep === STEPS.LOGIN) {
            getSuggestedLogins(dispatch, state, true)
                .then(() => {
                    changeStep(dispatch, nextStep);
                    dispatch(domikIsLoading(false));
                })
                .catch(() => dispatch(domikIsLoading(false)));
        } else {
            if (customs.isSkipEulaPage && nextStep === STEPS.EULA) {
                regAccount();
                return;
            }

            changeStep(dispatch, nextStep);
            dispatch(domikIsLoading(false));
        }
    }

    if ([STEPS.LOGIN, STEPS.PASSWORD].includes(step)) {
        changeStep(dispatch, nextStep);
        dispatch(domikIsLoading(false));
    }

    if (step === STEPS.EULA) {
        regAccount();
    }

    if (step === STEPS.SQ_SA) {
        if (states[FIELDS_NAMES.HINT_ANSWER] === 'valid') {
            changeStep(dispatch, nextStep);
            dispatch(domikIsLoading(false));
        } else {
            const {
                values: {hint_question_id, hint_question_custom, hint_answer}
            } = form;

            const validationPromise =
                hint_question_id === '99'
                    ? dispatch(
                          validateSecurityAnswer({hint_question_id, hint_question: hint_question_custom, hint_answer})
                      )
                    : dispatch(validateSecurityAnswer({hint_answer}));

            validationPromise
                .then(() => {
                    dispatch(setCaptchaRequired(true));
                })
                .catch(() => {
                    /** DO NOTHING */
                })
                .finally(() => {
                    dispatch(domikIsLoading(false));
                });
        }
    }
};

export const onBack = () => (dispatch, getState) => {
    const state = getState();
    const {userEntryFlow = {}, form = {}} = state;
    const {captchaRequired} = form;
    const {step} = userEntryFlow;
    const process = getProcessName(state);
    const isDoregishFlowAvailable = checkIsDoregishFlowProcess(state);
    const steps = getSteps(process, isDoregishFlowAvailable);
    const currentStep = step || steps[0];

    let nextStep = getPrevStep(currentStep, process, isDoregishFlowAvailable);

    if (nextStep === STEPS.ACCOUNTS) {
        nextStep = getPrevStep(nextStep, process, isDoregishFlowAvailable);
        dispatch(setAccountsSuggest([]));
    }

    metrika.send(`Клик "Назад" на экране ${currentStep}`);

    if (currentStep === STEPS.SQ_SA && captchaRequired) {
        dispatch(setCaptchaRequired(false));
        dispatch(updateStates({field: FIELDS_NAMES.HINT_ANSWER, status: ''}));

        return;
    }

    if (nextStep === STEPS.SQ_SA) {
        dispatch(setCaptchaRequired(true));
    }

    if ([STEPS.PHONE_CONFIRM, STEPS.PHONE, STEPS.PERSONAL_DATA_AND_PHONE].includes(nextStep)) {
        dispatch(domikIsLoading(true));

        api.userEntryFlowSubmit({process})
            .then(() => {
                changeStep(dispatch, steps[0]);
                dispatch(domikIsLoading(false));
            })
            .catch(() => dispatch(domikIsLoading(false)));

        return;
    }

    changeStep(dispatch, nextStep);
};

export const restartProcess = () => (dispatch, getState) => {
    const state = getState();
    const process = getProcessName(state);
    const isDoregishFlowAvailable = checkIsDoregishFlowProcess(state);
    const steps = getSteps(process, isDoregishFlowAvailable);

    metrika.send('Клик в "Попробовать еще раз"');

    api.userEntryFlowSubmit({process})
        .then(() => {
            dispatch(clearProcessError());
            dispatch(updateValues({field: FIELDS_NAMES.LOGIN, value: ''}));
            dispatch(updateValues({field: FIELDS_NAMES.PASSWORD, value: ''}));
            changeStep(dispatch, steps[0]);
        })
        .catch(() => {
            metrika.send(`Ошибка перезапуска процесса no_track`);
            dispatch(setProcessError(getAdvancedError('global', 'global')));
            dispatch(domikIsLoading(false));
        });
};
