import api from '@blocks/api';
import {getTrackId} from '@blocks/selectors';
import {TRACK_TYPES} from '@blocks/actions/tracks';
import {saveActionForRepeat} from '@blocks/common/actions';
import {
    setLoading,
    setError as setErrorAction,
    setSocial as setSocialAction,
    addSocial as addSocialAction,
    removeSocial as removeSocialAction,
    setAcceptVerification as setAcceptVerificationAction,
    setAcceptTerms as setAcceptTermsAction,
    setAcceptDelete as setAcceptDeleteAction,
    setPhotoSrc as setPhotoSrcAction,
    setEmail as setEmailAction,
    setEmailConfirmed,
    setEmailConfirmInProgress as setEmailConfirmInProgressAction,
    setPhone as setPhoneAction,
    setPhoneConfirmed,
    setPhoneConfirmInProgress as setPhoneConfirmInProgressAction,
    setReason as setReasonAction
} from './actions';

export const makeRequest = (dispatch, method, data = {}, isFormData) => {
    dispatch(setLoading(true));
    return api
        .request(method, data, isFormData && {processData: false, contentType: false})
        .always(() => dispatch(setLoading(false)));
};

export const setError = (error) => (dispatch) => {
    if (error && error.code === 'password.required') {
        return null;
    }
    dispatch(setErrorAction(error));
};

export const setSocial = (social, index) => (dispatch) => dispatch(setSocialAction({social, index}));
export const addSocial = () => (dispatch) => dispatch(addSocialAction());
export const removeSocial = (index) => (dispatch) => dispatch(removeSocialAction({index}));

export const setAcceptVerification = (value) => (dispatch) => dispatch(setAcceptVerificationAction({value}));
export const setAcceptTerms = (value) => (dispatch) => dispatch(setAcceptTermsAction({value}));
export const setAcceptDelete = (value) => (dispatch) => dispatch(setAcceptDeleteAction({value}));

export const setPhotoSrc = (src) => (dispatch) => dispatch(setPhotoSrcAction({src}));
export const postPhoto = (files) => (dispatch) => {
    const file = files[0];

    if (7 * 1024 * 1024 > file.size) {
        const reader = new FileReader();

        reader.readAsDataURL(file);
        reader.onload = () => dispatch(setPhotoSrcAction({src: reader.result}));
    } else {
        dispatch(setError({source: 'photo'}));
    }
};

export const setEmail = (value) => (dispatch, getState) => {
    const {verification: {error} = {}} = getState();

    if (error && error.source === 'email_button') {
        dispatch(setErrorAction(null));
    }
    dispatch(setEmailAction({value}));
};

export const setEmailAndConfirm = (value) => (dispatch) => {
    dispatch(setEmailAction({value}));
    dispatch(setEmailConfirmed());
};

export const postCreateEmail = (onDone, onFail) => (dispatch, getState) => {
    const state = getState();
    const {verification: {email: {value} = {}, error} = {}, common = {}, settings = {}} = state;
    const data = {
        email: value,
        retpath: common.retpath || common.profilePage,
        language: settings.language,
        uid: common.uid,
        code_only: true,
        track_id: getTrackId(state, TRACK_TYPES.COMMON),
        validator_ui_url: common.emailValidatorUiUrl
    };

    if (error && error.source === 'email_button') {
        dispatch(setErrorAction(null));
    }

    dispatch(saveActionForRepeat(postCreateEmail, onDone, onFail));

    return makeRequest(dispatch, 'email/send-confirmation-email', data)
        .done(onDone)
        .fail(onFail);
};

export const postEmailCommit = (code) => (dispatch, getState) => {
    const data = {
        track_id: getTrackId(getState(), TRACK_TYPES.COMMON),
        key: code ? code.trim() : ''
    };

    dispatch(saveActionForRepeat(postEmailCommit, code));

    return makeRequest(dispatch, 'email/confirm-by-code', data).done(() => dispatch(setEmailConfirmed()));
};

export const setEmailConfirmInProgress = () => (dispatch) => dispatch(setEmailConfirmInProgressAction());

export const setPhone = (value) => (dispatch, getState) => {
    const {verification: {error} = {}} = getState();

    if (error && error.source === 'phone_button') {
        dispatch(setErrorAction(null));
    }
    dispatch(setPhoneAction({value}));
};

export const postCreatePhone = (onDone, onFail) => (dispatch, getState) => {
    const state = getState();
    const {verification: {phone: {value} = {}, error} = {}} = state;
    const data = {
        track_id: getTrackId(state, TRACK_TYPES.COMMON),
        mode: 'confirmAndBindSecure',
        confirmMethod: 'by_sms',
        checkCaptcha: false,
        number: value
    };

    if (error && error.source === 'phone_button') {
        dispatch(setErrorAction(null));
    }

    dispatch(saveActionForRepeat(postCreatePhone, onDone, onFail));

    return makeRequest(dispatch, 'phone-confirm-code-submit', data)
        .done(onDone)
        .fail(onFail);
};

export const postPhoneCommit = (code, onDone, onFail) => (dispatch, getState) => {
    const data = {
        track_id: getTrackId(getState(), TRACK_TYPES.COMMON),
        code: code ? code.trim() : '',
        mode: 'confirmAndBindSecure'
    };

    dispatch(saveActionForRepeat(postPhoneCommit, code, onDone, onFail));

    return makeRequest(dispatch, 'phone-confirm-code', data)
        .done(() => {
            dispatch(setPhoneConfirmed());
            onDone && onDone();
        })
        .fail(onFail);
};

export const setPhoneConfirmInProgress = () => (dispatch) => dispatch(setPhoneConfirmInProgressAction());

export const setReason = (reason) => (dispatch) => dispatch(setReasonAction(reason));

export const postForm = () => (dispatch, getState) => {
    const state = getState() || {};
    const {verification, common} = state || {};
    const {socials = [], reason = ''} = verification || {};
    const {csrf} = common || {};
    const attach = document.getElementById('VerificationAttach') || {files: []};
    const file = attach.files[0];

    const data = new FormData();

    data.append('additional_info', (reason || '').trim());
    data.append('attach_file', file);
    data.append('track_id', getTrackId(state, TRACK_TYPES.COMMON));
    data.append('csrf_token', csrf);
    socials.filter(Boolean).forEach((social, index) => data.append(`social${index}`, (social || '').trim()));

    return makeRequest(dispatch, '/profile/verification', data, true);
};

export const postFormPpl = () => (dispatch, getState) => {
    const state = getState() || {};
    const {verification, common} = state || {};
    const {reason = '', checkboxes = {}, serpUri = '', email = {}, socials = []} = verification || {};
    const {isAcceptDelete} = checkboxes;
    const {csrf} = common || {};
    const {value: emailValue} = email;
    const attach = document.getElementById('VerificationAttach') || {files: []};
    const file = attach.files[0];

    const data = new FormData();

    data.append('additional_info', (reason || '').trim());
    data.append('require_delete', Number(isAcceptDelete).toString());
    data.append('attach_file', file);
    data.append('track_id', getTrackId(state, TRACK_TYPES.COMMON));
    data.append('csrf_token', csrf);
    data.append('serp_uri', serpUri);
    data.append('default_email', emailValue);
    socials.filter(Boolean).forEach((social, index) => data.append(`social${index}`, (social || '').trim()));

    return makeRequest(dispatch, '/profile/ppl', data, true);
};
