import api from '@blocks/api';
import {getLanguage, getTrackId, getFormValues, getFormErrors} from '@blocks/selectors';
import {updateErrors, updateValues} from '@blocks/actions/form';
import {changeLoadingStatus} from '@blocks/actions/spin';
import reloadCaptcha from '@components/Captcha/actions/reloadCaptcha';
import {getError} from '@blocks/utils';
import {FIELDS_NAMES} from '@components/Field/names';

export const CHANGE_PWD_INIT = 'CHANGE_PWD_INIT';
export const CHANGE_PWD_DONE = 'CHANGE_PWD_DONE';
export const CHANGE_PWD_GLOGOUT = 'CHANGE_PWD_GLOGOUT';

export function changePasswordInit(payload) {
    return {
        type: CHANGE_PWD_INIT,
        revokers: payload
    };
}

export function setPasswordChanged() {
    return {
        type: CHANGE_PWD_DONE
    };
}

export function toggleGlobalLogout(payload) {
    return {
        type: CHANGE_PWD_GLOGOUT,
        payload
    };
}

export function changePasswordSubmit() {
    return (dispatch, getState) => {
        const state = getState();
        const {common} = state;
        const {retpath, isPDD} = common;
        const data = {
            is_pdd: isPDD,
            track_id: getTrackId(state, 'common'),
            retpath,
            lang: getLanguage(state)
        };

        api.request('change-password-submit', data)
            .done((response = {}) => {
                if (response.status === 'ok' && response.revokers) {
                    dispatch(changePasswordInit(response.revokers));
                    return;
                }

                dispatch(updateErrors({field: 'password', error: getError('password', 'global')}));
            })
            .fail((error = {}) => {
                const errorCode = error.error && error.error[0];

                dispatch(updateErrors({field: 'password', error: getError('password', errorCode)}));
            });
    };
}

export function changePasswordCommit() {
    return (dispatch, getState) => {
        const state = getState();
        const {common, changePassword} = state;
        const form = getFormValues(state);
        const {password: passwordError, password_confirm: confirmError} = getFormErrors(state);
        const {password, current_password, captcha} = form;
        const {isPDD} = common;
        const {isGlobalLogoutChecked} = changePassword;
        const data = {
            answer: captcha,
            current_password,
            password,
            track_id: getTrackId(state, 'common'),
            is_pdd: isPDD,
            revoke_web_sessions: isGlobalLogoutChecked,
            revoke_tokens: isGlobalLogoutChecked,
            revoke_app_passwords: isGlobalLogoutChecked
        };

        dispatch(updateErrors({field: 'captcha', error: {}}));

        if (!checkFormIsFilled(data, dispatch)) {
            return;
        }

        if (passwordError.code || confirmError.code) {
            return;
        }

        dispatch(changeLoadingStatus(true));

        api.request('change-password-commit', data)
            .done((response = {}) => {
                if (response.status === 'ok') {
                    dispatch(setPasswordChanged());
                    return;
                }

                dispatch(updateErrors({field: 'password', error: getError('password', 'global')}));
            })
            .fail((error = {}) => {
                if (Array.isArray(error.error)) {
                    const filteredError = error.error.filter((code) => code !== 'captcha.required');
                    const errorCode = filteredError[0];

                    dispatch(reloadCaptcha());
                    dispatch(updateValues({field: FIELDS_NAMES.CAPTCHA, value: ''}));

                    if (errorCode === 'captcha.cannot_locate') {
                        return dispatch(
                            updateErrors({field: 'captcha', error: getError('captcha', 'captcha.not_matched')})
                        );
                    }

                    if (errorCode === 'password.not_matched') {
                        return dispatch(
                            updateErrors({field: 'current_password', error: getError('password', 'not_matched')})
                        );
                    }

                    return dispatch(updateErrors({field: 'password', error: getError('password', errorCode)}));
                }

                dispatch(updateErrors({field: 'password', error: getError('password', 'global')}));
            })
            .always(() => dispatch(changeLoadingStatus(false)));
    };
}

function checkFormIsFilled(data = {}, dispatch) {
    const hasValue = (value) => value !== undefined && value !== '';

    let isFormFilled = true;

    if (!dispatch) {
        return false;
    }

    for (const key in data) {
        if (!hasValue(data[key])) {
            const errorField = key === 'answer' ? 'captcha' : key;

            isFormFilled = false;
            dispatch(updateErrors({field: errorField, error: getError(errorField, 'missingvalue')}));
        }
    }

    return isFormFilled;
}
