import api from '@blocks/api';
import {getTrackId, getFormValues} from '@blocks/selectors';
import {TRACK_TYPES} from '@blocks/actions/tracks';
import {updateErrors} from '@blocks/actions/form';
import {getError} from '@blocks/utils';
import passport from '@plibs/pclientjs';
import {saveActionForRepeat} from '@blocks/common/actions';
import {changeLoadingStatus} from '@blocks/actions/spin';

export const CONFIRMATION_CODE_SENT = 'CONFIRMATION_CODE_SENT';
export const CONFIRMATION_CODE_CHECKED = 'CONFIRMATION_CODE_CHECKED';
export const CLEAR_EMAIL_EDIT_STATE = 'CLEAR_EMAIL_EDIT_STATE';
export const MARK_EMAIL_DELETED = 'MARK_EMAIL_DELETED';
export const UNMARK_EMAIL_DELETED = 'UNMARK_EMAIL_DELETED';
export const SET_DELETE_DONE = 'SET_DELETE_DONE';

export function setConfirmationCodeSent() {
    return {
        type: CONFIRMATION_CODE_SENT
    };
}

export function setConfirmationCodeChecked() {
    return {
        type: CONFIRMATION_CODE_CHECKED
    };
}

export function clearEmailEditState() {
    return {
        type: CLEAR_EMAIL_EDIT_STATE
    };
}

export function markEmailDeleted(email) {
    return {
        type: MARK_EMAIL_DELETED,
        email
    };
}

export function unmarkEmailDeleted(email) {
    return {
        type: UNMARK_EMAIL_DELETED,
        email
    };
}

export function setDeleteDone() {
    return {
        type: SET_DELETE_DONE
    };
}

export function sendConfirmationCode() {
    return (dispatch, getState) => {
        const state = getState();
        const form = getFormValues(state);
        const {common = {}, settings = {}} = state;
        const data = {
            email: form.email,
            retpath: common.retpath || common.profilePage,
            language: settings.language,
            uid: common.uid,
            track_id: getTrackId(state, TRACK_TYPES.COMMON),
            validator_ui_url: `https://${location.hostname}${passport.basePath}email/confirm-by-link`
        };

        dispatch(changeLoadingStatus(true));
        dispatch(saveActionForRepeat(sendConfirmationCode, {email: form.email}));
        api.request('email/send-confirmation-email', data)
            .done((response = {}) => {
                if (response.status === 'ok') {
                    return dispatch(setConfirmationCodeSent());
                }

                dispatch(updateErrors({field: 'email', error: getError('email', 'global')}));
            })
            .fail((error = {}) => {
                const errorCode = Array.isArray(error.errors) ? error.errors[0] : 'global';

                if (errorCode === 'password.required') {
                    return;
                }

                dispatch(updateErrors({field: 'email', error: getError('email', errorCode)}));
            })
            .always(() => dispatch(changeLoadingStatus(false)));
    };
}

export function checkConfirmationCode() {
    return (dispatch, getState) => {
        const state = getState();
        const form = getFormValues(state);
        const data = {
            track_id: getTrackId(state, TRACK_TYPES.COMMON),
            key: form.email_code ? form.email_code.trim() : ''
        };

        dispatch(changeLoadingStatus(true));
        api.request('email/confirm-by-code', data)
            .done((response = {}) => {
                if (response.status === 'ok') {
                    return dispatch(setConfirmationCodeChecked());
                }

                dispatch(updateErrors({field: 'email_code', error: getError('email_code', 'global')}));
            })
            .fail((error = {}) => {
                const errorCode = Array.isArray(error.errors) ? error.errors[0] : 'global';

                dispatch(updateErrors({field: 'email_code', error: getError('email_code', errorCode)}));
            })
            .always(() => dispatch(changeLoadingStatus(false)));
    };
}

export function deleteEmail() {
    return (dispatch, getState) => {
        const state = getState();
        const {editEmails = {}} = state;
        const {emailsToDelete = []} = editEmails;
        const deleteRequests = emailsToDelete.map((email) =>
            api.request('email/delete', {track_id: getTrackId(state, TRACK_TYPES.COMMON), email})
        );

        dispatch(changeLoadingStatus(true));
        dispatch(saveActionForRepeat(deleteEmail));

        Promise.all(deleteRequests)
            .then((response = []) => {
                if (response[0].status === 'ok') {
                    dispatch(setDeleteDone());
                }

                dispatch(changeLoadingStatus(false));
            })
            .catch((error) => {
                const errorCode = Array.isArray(error.errors) ? error.errors[0] : 'global';

                dispatch(changeLoadingStatus(false));

                if (errorCode === 'password.required') {
                    return;
                }

                dispatch(updateErrors({field: 'email', error: getError('email', errorCode)}));
            });
    };
}
