import {push} from 'react-router-redux';
import {getFormValues, getRedirectUrl, getTrackId, isKeyEnabled2FASelector} from '@blocks/selectors';
import {domikIsLoading} from '@blocks/auth/actions';
import {SUCCESS_AUTH_GOAL, SUCCESS_AUTH_BY} from '@blocks/authv2/metrics_constants';
import {changePagePopupType, changePagePopupVisibility} from '@blocks/authv2/actions';
import {setUserEntryProcessedUid, setProcessError} from '@blocks/UserEntryFlow/actions';
import {retryToRequestPhoneConfirmationCode} from '@blocks/actions/phoneConfirm';
import {TRACK_TYPES, updateTrackId} from '@blocks/actions/tracks';
import {updateErrors} from '@blocks/actions/form';
import {FIELDS_NAMES} from '@components/Field/names';
import {getAdvancedError, getError} from '@blocks/utils';
import metrics from '@blocks/metrics';
import api from '@blocks/api';
import {amSetAnalyticsToRetpath} from '@blocks/authv2/actions/nativeMobileApi';
import sendToForcePasswordChange from '@blocks/authv2/actions/forcePasswordChange/sendToForcePasswordChange';
import {updatePhoneStatus} from '@blocks/authv2/actions/forcePasswordChange';
import {checkIsDoregishFlowProcess} from '@blocks/UserEntryFlow/utils';
import {ACCOUNT_TYPES, getAccountTypeByAlias} from '@blocks/authv2/utils/accountTypes';
import {authEnableKey2FA} from '@blocks/authv2/actions/authEnableKey2FA';

export const neoPhonishAuth = ({trackId, firstname, lastname, uid, accountTypeAlias = ''}) => (dispatch, getState) => {
    const state = getState();
    const {am: {isAm} = {}, common = {}} = state;
    const {preRegCompleteUrl} = common;
    const isDoregishFlowAvailable = checkIsDoregishFlowProcess(state);
    const handle = isDoregishFlowAvailable ? 'neoPhonishRestore' : 'neoPhonishAuth';
    const params = {
        trackId,
        uid
    };
    const accountType = getAccountTypeByAlias(accountTypeAlias) || 'neophonish';
    const isKeyEnabled2FA = isKeyEnabled2FASelector(state);

    let redirectUrl = getRedirectUrl(state);

    if (isAm) {
        dispatch(amSetAnalyticsToRetpath('auth_neo_phonish'));
        redirectUrl = getRedirectUrl(getState());
    }

    dispatch(domikIsLoading(true));

    if (handle === 'neoPhonishAuth') {
        params.firstname = firstname;
        params.lastname = lastname;
        params.retpath = redirectUrl;
    }

    api[handle](params).then((response) => {
        const {status, redirectUrl: redirectUrlFromResponse, state, retpath} = response;

        if (status === 'ok') {
            const handleFinish = async () => {
                if (isKeyEnabled2FA) {
                    if (accountType !== ACCOUNT_TYPES.PORTAL) {
                        return dispatch(push(preRegCompleteUrl));
                    }
                    await dispatch(authEnableKey2FA());
                }

                if (state === 'change_password') {
                    dispatch(updatePhoneStatus('phone-confirmed'));
                    dispatch(updateTrackId({id: trackId, type: TRACK_TYPES.COMMON}));
                    dispatch(sendToForcePasswordChange());
                    return;
                }

                if (state && redirectUrlFromResponse) {
                    return (window.location = redirectUrlFromResponse);
                }

                metrics.goal(SUCCESS_AUTH_GOAL);
                metrics.send([`${SUCCESS_AUTH_BY}${accountType}`]);

                return (window.location = retpath || redirectUrl);
            };

            if (isAm) {
                dispatch(domikIsLoading(false));

                // В интеграции веб ам есть нативный спиннер
                // чтобы они не пересекались с вебным, нужно
                // скрыть вебный и дать реакту отрисовать его скрытие
                return setTimeout(() => {
                    handleFinish();
                });
            }

            return handleFinish();
        }

        const {errors = []} = response;
        const error = errors[0];

        // нужно подтвердить телефон СМСкой
        if (error === 'phone.not_confirmed') {
            return api.getTrack({trackId, type: TRACK_TYPES.AUTH}).then((response = {}) => {
                const {id} = response;

                if (id) {
                    dispatch(retryToRequestPhoneConfirmationCode(id));
                    dispatch(changePagePopupVisibility(true));
                    dispatch(setUserEntryProcessedUid(uid));
                    dispatch(changePagePopupType('neoPhonishAuthConfirmationBySMS'));
                } else {
                    dispatch(setProcessError(getAdvancedError('global', 'global')));
                }

                dispatch(domikIsLoading(false));
            });
        }

        dispatch(changePagePopupVisibility(false));
        dispatch(setProcessError(getAdvancedError('neoPhonishAuth', error)));
        dispatch(domikIsLoading(false));
    });
};

export const commitNeoPhonishAuth = () => (dispatch, getState) => {
    const state = getState();
    const formValues = getFormValues(state);
    const {userEntryFlow = {}} = state;
    const {processedUid: uid} = userEntryFlow;
    const {lastname, firstname, phoneCode: code} = formValues;
    const trackId = getTrackId(state, TRACK_TYPES.AUTH);

    dispatch(domikIsLoading(true));

    const errorCallback = (error = {}) => {
        const {errors = []} = error;
        const code = errors[0] || 'global';

        dispatch(
            updateErrors({field: FIELDS_NAMES.PHONE_CODE, error: code ? getError(FIELDS_NAMES.PHONE_CODE, code) : {}})
        );
        dispatch(domikIsLoading(false));
    };

    api.checkPhoneConfirmationCode({trackId, code})
        .then((response = {}) => {
            const {status} = response;

            if (status === 'ok') {
                return dispatch(neoPhonishAuth({trackId, firstname, lastname, uid}));
            }

            errorCallback(response);
        })
        .catch(errorCallback);
};
