import {changeLoadingStatus} from '@blocks/actions/spin';
import api from '@blocks/api';
import {getTrackId} from '@blocks/selectors';
import {TRACK_TYPES} from '@blocks/actions/tracks';
import {updateErrors} from '@blocks/actions/form';
import {FIELDS_NAMES} from '@components/Field/names';
import {getError} from '@blocks/utils';

export const PHONE_PLACEHOLDERS = {
    by: '375 xx xxx xx xx',
    com: '1 xxx xxx xx xx',
    ru: '+7 (000) 000-00-00',
    'com.tr': '90 xxx xxx xx xx',
    uz: '998 xx xxx xx xx',
    kz: '77 xx xxx xx xx',
    ua: '380 xxx xx xx xx'
};

export const UPDATE_PHONE_PROCESS_INFO = 'UPDATE_PHONE_PROCESS_INFO';
export const SET_PHONE_PROCESS_CODE_SEND = 'SET_PHONE_PROCESS_CODE_SEND';
export const SET_PHONE_PROCESS_CODE_CHECKED = 'SET_PHONE_PROCESS_CODE_CHECKED';
export const SET_PHONE_PROCESS_TYPE = 'SET_PHONE_PROCESS_TYPE';
export const SET_PHONE_REPLACE_DONE = 'SET_PHONE_REPLACE_DONE';
export const SET_PHONE_REPLACE_STARTED = 'SET_PHONE_REPLACE_STARTED';
export const TOGGLE_PHONE_UNAVAILABLE = 'TOGGLE_PHONE_UNAVAILABLE';

export function setPhoneProcessType(payload) {
    return {
        type: SET_PHONE_PROCESS_TYPE,
        payload
    };
}

export function updatePhoneInfo(payload) {
    return {
        type: UPDATE_PHONE_PROCESS_INFO,
        payload
    };
}

export function setPhoneCodeSent() {
    return {
        type: SET_PHONE_PROCESS_CODE_SEND
    };
}

export function setPhoneCodeChecked() {
    return {
        type: SET_PHONE_PROCESS_CODE_CHECKED
    };
}

export function setPhoneReplaceStarted() {
    return {
        type: SET_PHONE_REPLACE_STARTED
    };
}

export function setPhoneReplaceDone() {
    return {
        type: SET_PHONE_REPLACE_DONE
    };
}

export function togglePhoneAvailable(payload = false) {
    return {
        type: TOGGLE_PHONE_UNAVAILABLE,
        isOldNumberUnAvailable: payload
    };
}

export function getPhoneState() {
    return (dispatch, getState) => {
        const state = getState();

        dispatch(changeLoadingStatus(true));

        api.request('yasms.state', {track_id: getTrackId(state, TRACK_TYPES.COMMON)})
            .done((response = {}) => {
                if (response.status === 'ok') {
                    dispatch(updatePhoneInfo(parsePhones(response.account)));
                    return;
                }

                dispatch(updateErrors({field: FIELDS_NAMES.PHONE, error: getError(FIELDS_NAMES.PHONE, 'global')}));
            })
            .fail((error = {}) => {
                const errorCode = Array.isArray(error.errors) ? error.errors[0] : 'global';

                dispatch(updateErrors({field: FIELDS_NAMES.PHONE, error: getError(FIELDS_NAMES.PHONE, errorCode)}));
            })
            .always(() => dispatch(changeLoadingStatus(false)));
    };
}

function parsePhones(data) {
    const phonesList = [];

    if (!data.phones) {
        return;
    }

    for (const phone in data.phones) {
        if (!data.phones.hasOwnProperty(phone)) {
            continue;
        }

        phonesList.push(getPhoneInfo(data.phones[phone]));
    }
    const availablePhone = phonesList.find((phone) => phone.isInUse) || {};
    const isInQuarantine = phonesList.some((phone) => phone.isInQuarantine);
    const quarantineDaysLeft = isInQuarantine ? phonesList.find((phone) => phone.isInQuarantine).quarantineEnd : null;

    return {
        isInProgress: Boolean(availablePhone.operationId) && !isInQuarantine,
        isInQuarantine,
        quarantineDaysLeft,
        operationId: availablePhone.operationId,
        number: availablePhone.number,
        phonesList
    };
}

function getPhoneInfo(phone) {
    const {operation} = phone;
    const phoneInfo = {
        id: phone.id,
        number: phone.number,
        isSecured: Boolean(phone.secured),
        isBound: Boolean(phone.bound)
    };

    if (operation) {
        phoneInfo.isInUse = operation.does_user_admit_phone;
        phoneInfo.operationType = operation.type;
        phoneInfo.operationId = operation.id;
        phoneInfo.isInQuarantine = operation.in_quarantine;
        phoneInfo.quarantineEnd = Math.ceil(
            (new Date(operation.finished * 1000).getTime() - Date.now()) / 1000 / 60 / 60 / 24
        );
    }

    return phoneInfo;
}
