import api from '@blocks/api';
import {TRACK_TYPES} from '@blocks/actions/tracks';
import {getTrackId, getLanguage, getFormValues} from '@blocks/selectors';
import {getError} from '@blocks/utils';
import {updateErrors} from '../form';
import {changeLoadingStatus} from '@blocks/actions/spin';
import {FIELDS_NAMES} from '@components/Field/names';
import {finishPhoneBind} from '@blocks/actions/phone/phoneBind';
import {confirmReplace, updateReplaceInfo} from '@blocks/actions/phone/phoneReplace';
import {setPhoneCodeSent} from '@blocks/actions/phone';
import {saveActionForRepeat} from '@blocks/common/actions';

export function resendCode(id) {
    return function(dispatch, getState) {
        const state = getState();
        const {phoneProcess = {}} = state;
        const {operationId} = phoneProcess;
        const params = {
            track_id: getTrackId(state, TRACK_TYPES.COMMON),
            display_language: getLanguage(state),
            id: id || operationId
        };

        dispatch(changeLoadingStatus(true));

        api.request('yasms.code.resend', params)
            .done((response) => {
                const {status} = response;

                if (status === 'ok') {
                    dispatch(setPhoneCodeSent());
                    return;
                }

                dispatch(updateErrors({field: FIELDS_NAMES.PHONE, error: getError(FIELDS_NAMES.PHONE, 'global')}));
            })
            .fail((error) => {
                const errorCode = error.error && error.error[0];

                dispatch(updateErrors({field: FIELDS_NAMES.PHONE, error: getError(FIELDS_NAMES.PHONE, errorCode)}));
            })
            .always(() => dispatch(changeLoadingStatus(false)));
    };
}

export function confirmCode() {
    return (dispatch, getState) => {
        const state = getState();
        const {phoneProcess = {}} = state;
        const formValues = getFormValues(state) || {};
        const {phoneCode} = formValues;
        const {operationId, processType, skipPasswordCheck} = phoneProcess;

        if (!phoneCode.trim()) {
            dispatch(
                updateErrors({
                    field: FIELDS_NAMES.PHONE_CODE,
                    error: getError(FIELDS_NAMES.PHONE_CODE, 'code.missingvalue')
                })
            );
            return;
        }

        dispatch(changeLoadingStatus(false));

        dispatch(saveActionForRepeat(confirmCode));

        api.request('yasms.code.check', {
            track_id: getTrackId(state, TRACK_TYPES.COMMON),
            operationId,
            code: phoneCode
        })
            .done((response = {}) => {
                if (response.status === 'ok') {
                    if (processType === 'replace' && skipPasswordCheck) {
                        return dispatch(confirmReplace());
                    }

                    return dispatch(checkPassword());
                }
                dispatch(updateErrors({field: FIELDS_NAMES.PHONE, error: getError(FIELDS_NAMES.PHONE, 'global')}));
            })
            .fail((error = {}) => {
                const errorCode = Array.isArray(error.errors) ? error.errors[0] : 'global';

                dispatch(
                    updateErrors({
                        field: FIELDS_NAMES.PHONE_CODE,
                        error: getError(FIELDS_NAMES.PHONE_CODE, errorCode)
                    })
                );
                dispatch(changeLoadingStatus(false));
            });
    };
}

export function checkPassword() {
    return (dispatch, getState) => {
        const state = getState();
        const {phoneProcess, hasPasswordChecked} = state;
        const formValues = getFormValues(state) || {};
        const {password} = formValues;
        const {isOldNumberUnAvailable, processType, operationId} = phoneProcess;
        const onPasswordSuccess = processType === 'replace' ? confirmReplace : finishPhoneBind;

        if (hasPasswordChecked) {
            return dispatch(onPasswordSuccess());
        }

        if (!password.trim()) {
            dispatch(
                updateErrors({field: FIELDS_NAMES.PASSWORD, error: getError(FIELDS_NAMES.PASSWORD, 'missingvalue')})
            );
            dispatch(changeLoadingStatus(false));
            return;
        }
        api.request('yasms.password.check', {
            track_id: getTrackId(state, TRACK_TYPES.COMMON),
            operationId,
            password
        })
            .done((response = {}) => {
                if (response.status === 'ok') {
                    if (!isOldNumberUnAvailable && processType === 'replace') {
                        dispatch(updateReplaceInfo());
                        return;
                    }

                    dispatch(onPasswordSuccess());
                    return;
                }
                dispatch(
                    updateErrors({
                        field: FIELDS_NAMES.PASSWORD,
                        error: getError(FIELDS_NAMES.PASSWORD, 'global')
                    })
                );
            })
            .fail((error = {}) => {
                const errorCode = Array.isArray(error.errors) ? error.errors[0] : 'global';

                dispatch(
                    updateErrors({
                        field: FIELDS_NAMES.PASSWORD,
                        error: getError(FIELDS_NAMES.PASSWORD, errorCode)
                    })
                );
                dispatch(changeLoadingStatus(false));
            });
    };
}

export function cancelProcess() {
    return function(dispatch, getState) {
        const state = getState();
        const {phoneProcess = {}} = state;

        dispatch(changeLoadingStatus(true));

        api.request('yasms.operation.cancel', {
            track_id: getTrackId(state, TRACK_TYPES.COMMON),
            id: phoneProcess.operationId
        })
            .done((response) => {
                if (response.status !== 'ok') {
                    dispatch(updateErrors({field: FIELDS_NAMES.PHONE, error: getError(FIELDS_NAMES.PHONE, 'global')}));
                }
            })
            .fail((error) => {
                const errorCode = error.error && error.error[0];

                dispatch(updateErrors({field: FIELDS_NAMES.PHONE, error: getError(FIELDS_NAMES.PHONE, errorCode)}));
            })
            .always(() => dispatch(changeLoadingStatus(false)));
    };
}
