import api from '@blocks/api';
import {TRACK_TYPES} from '@blocks/actions/tracks';
import {getTrackId, getLanguage} from '@blocks/selectors';
import {getError} from '@blocks/utils';
import {setPhoneCodeSent, updateErrors, updateValues} from '../form';
import {changeLoadingStatus} from '@blocks/actions/spin';
import {FIELDS_NAMES} from '@components/Field/names';
import {getPhoneState, updatePhoneInfo, setPhoneReplaceStarted, setPhoneReplaceDone} from '@blocks/actions/phone';

export function sendCode(number) {
    return function(dispatch, getState) {
        const state = getState();
        const {settings = {}, phoneProcess = {}} = state;
        const {isOldNumberUnAvailable} = phoneProcess;
        const params = {
            track_id: getTrackId(state, TRACK_TYPES.COMMON),
            display_language: getLanguage(state),
            owner: !isOldNumberUnAvailable,
            number,
            country: settings.country
        };

        dispatch(changeLoadingStatus(true));

        api.request('yasms.secure.replace.submit', params)
            .done((response) => {
                if (response.status === 'ok') {
                    dispatch(setPhoneReplaceStarted());
                    dispatch(getPhoneState());
                    return;
                }

                dispatch(updateErrors({field: FIELDS_NAMES.PHONE, error: getError(FIELDS_NAMES.PHONE, 'global')}));
            })
            .fail((error) => {
                const errorCode = error.errors && error.errors[0];

                dispatch(changeLoadingStatus(false));
                dispatch(updateErrors({field: FIELDS_NAMES.PHONE, error: getError(FIELDS_NAMES.PHONE, errorCode)}));
            });
    };
}

export function resendCode() {
    return function(dispatch, getState) {
        const state = getState();
        const {phoneReplace = {}} = state;
        const {operation, state: phoneReplaceState} = phoneReplace;
        const params = {
            track_id: getTrackId(state, TRACK_TYPES.COMMON),
            display_language: getLanguage(state),
            id: phoneReplaceState.operationId || operation.id
        };

        dispatch(changeLoadingStatus(true));

        api.request('yasms.code.resend', params)
            .done((response) => {
                const {status} = response;

                if (status === 'ok') {
                    dispatch(setPhoneCodeSent());
                    return;
                }

                dispatch(updateErrors({field: FIELDS_NAMES.PHONE, error: getError(FIELDS_NAMES.PHONE, 'global')}));
            })
            .fail((error) => {
                const errorCode = error.error && error.error[0];

                dispatch(updateErrors({field: FIELDS_NAMES.PHONE, error: getError(FIELDS_NAMES.PHONE, errorCode)}));
            })
            .always(() => dispatch(changeLoadingStatus(false)));
    };
}

export function confirmReplace() {
    return (dispatch, getState) => {
        const state = getState();
        const {phoneProcess = {}} = state;
        const {operationId} = phoneProcess;

        dispatch(changeLoadingStatus(true));
        api.request('yasms.secure.replace.commit', {
            id: operationId,
            track_id: getTrackId(state, TRACK_TYPES.COMMON)
        })
            .done((response = {}) => {
                if (response.status === 'ok') {
                    return dispatch(setPhoneReplaceDone(true));
                }

                dispatch(
                    updateErrors({field: FIELDS_NAMES.PHONE_CODE, error: getError(FIELDS_NAMES.PHONE_CODE, 'global')})
                );
            })
            .fail((error = {}) => {
                const errorCode = Array.isArray(error.errors) ? error.errors[0] : 'global';

                dispatch(
                    updateErrors({field: FIELDS_NAMES.PHONE_CODE, error: getError(FIELDS_NAMES.PHONE_CODE, errorCode)})
                );
            })
            .always(() => dispatch(changeLoadingStatus(false)));
    };
}

export function updateReplaceInfo() {
    return (dispatch, getState) => {
        const {phoneProcess} = getState();
        const {phonesList} = phoneProcess;
        const newPhone = phonesList.find((phone) => phone.operationType === 'bind');

        dispatch(
            updatePhoneInfo({
                skipPasswordCheck: true,
                operationId: newPhone.operationId,
                phoneId: newPhone.id,
                processedPhone: newPhone.number
            })
        );
        dispatch(updateValues({field: FIELDS_NAMES.PHONE_CODE, value: ''}));
    };
}
