import {changePagePopupType, changePagePopupVisibility, domikIsLoading} from '@blocks/authv2/actions';
import redirectToRetpath from '@blocks/authv2/actions/redirectToRetpath';
import checkCaptcha from '@blocks/authv2/actions/checkCaptcha';
import {updateErrors} from '@blocks/actions/form';
import {TRACK_TYPES} from '@blocks/actions/tracks';
import {getLanguage, getTrackId, getFormValues, isAndroidAm, isKeyEnabled2FASelector} from '@blocks/selectors';
import {FIELDS_NAMES} from '@components/Field/names';
import {getError, getUnixTimeStamp, sendMessage} from '@blocks/utils';
import metrika from '@blocks/metrics';
import api from '@blocks/api';
import {Platforms} from '@blocks/authv2/nativeMobileApi/constants';
import {amRequestSmsCode, amStorePhoneNumber} from '@blocks/authv2/actions/nativeMobileApi';
import {authEnableKey2FA} from '@blocks/authv2/actions/authEnableKey2FA';

const SUCCESS_MESSAGE = {phoneConfirm: 'success'};
const ERROR_MESSAGE = {phoneConfirm: 'error'};

export const UPDATE_PHONE_CONFIRM_NUMBER = 'UPDATE_PHONE_CONFIRM_NUMBER';
export const UPDATE_PHONE_CONFIRM_CODE = 'UPDATE_PHONE_CONFIRM_CODE';
export const SET_PHONE_CONFIRM_ERRORS = 'SET_PHONE_CONFIRM_ERRORS';
export const SET_PHONE_CONFIRM_TIMER = 'SET_PHONE_CONFIRM_TIMER';
export const CLEAR_SECURE_PHONE_NUMBER = 'CLEAR_SECURE_PHONE_NUMBER';
export const SET_PHONE_CONFIRM_METHOD = 'SET_PHONE_CONFIRM_METHOD';
export const SET_PHONE_CONFIRMATION_CALLING_NUMBER_TEMPLATE = 'SET_PHONE_CONFIRMATION_CALLING_NUMBER_TEMPLATE';
export const SET_PHONE_CONFIRMATION_PHONE_CHECK_STATE = 'SET_PHONE_CONFIRMATION_PHONE_CHECK_STATE';
export const SET_PHONE_CONFIRMATION_DENY_RESEND_UNTIL = 'SET_SIGN_UP_PHONE_CONFIRMATION_DENY_RESEND_UNTIL';
export const SET_PHONE_CONFIRMATION_INTERNATIONAL_PHONE_NUMBER = 'SET_PHONE_CONFIRMATION_INTERNATIONAL_PHONE_NUMBER';
export const SET_PHONE_CODE_TIMER = 'SET_PHONE_CODE_TIMER';
export const SET_CONFIRM_PROCESS_STATE = 'SET_CONFIRM_PROCESS_STATE';
export const SET_PREPARED_STORE = 'SET_PREPARED_STORE';
export const SET_IS_PASSWORD_REQUIRED = '[PHONE_CONFIRM] SET_IS_PASSWORD_REQUIRED';
export const SET_CAPTCHA_REQUIRED = '[PHONE_CONFIRM] SET_CAPTCHA_REQUIRED';

export const setPhoneConfirmationDenyResendUntilTime = (denyResendUntil) => (dispatch) =>
    dispatch({
        type: SET_PHONE_CONFIRMATION_DENY_RESEND_UNTIL,
        payload: {denyResendUntil, isConfirmationCodeSent: true}
    });

export const setPhoneConfirmationInternationalPhoneNumber = (internationalPhoneNumber) => ({
    type: SET_PHONE_CONFIRMATION_INTERNATIONAL_PHONE_NUMBER,
    payload: {internationalPhoneNumber}
});

export const setPhoneConfirmMethod = (phoneConfirmMethod = 'by_sms') => (dispatch) =>
    dispatch({type: SET_PHONE_CONFIRM_METHOD, payload: {phoneConfirmMethod}});

export const setPhoneConfirmationCallingNumberTemplate = (callingNumberTemplate = '') => (dispatch) =>
    dispatch({type: SET_PHONE_CONFIRMATION_CALLING_NUMBER_TEMPLATE, payload: {callingNumberTemplate}});

export const setPhoneConfirmationPhoneCheckState = (shouldCheckPhone = false) => (dispatch) =>
    dispatch({type: SET_PHONE_CONFIRMATION_PHONE_CHECK_STATE, payload: {shouldCheckPhone}});

export const setErrors = (errors) => (dispatch) => dispatch({type: SET_PHONE_CONFIRM_ERRORS, payload: {errors}});

export const updatePhoneNumber = (phoneNumber) => (dispatch) =>
    dispatch({type: UPDATE_PHONE_CONFIRM_NUMBER, payload: {phoneNumber}});

export const updateConfirmationCode = (confirmationCode) => (dispatch) =>
    dispatch({type: UPDATE_PHONE_CONFIRM_CODE, payload: {confirmationCode}});

export const clearSecurePhoneNumber = () => (dispatch) => dispatch({type: CLEAR_SECURE_PHONE_NUMBER});

export const setPreparedStore = (payload) => ({
    type: SET_PREPARED_STORE,
    payload
});

export const setConfirmProcessState = (isConfirmProcess) => (dispatch) =>
    dispatch({type: SET_CONFIRM_PROCESS_STATE, payload: {isConfirmProcess}});

const setCaptchaRequired = (isCaptchaRequired) => (dispatch) =>
    dispatch({type: SET_CAPTCHA_REQUIRED, payload: {isCaptchaRequired}});

export function preparePhoneConfirmStore() {
    return async (dispatch, getState) => {
        const state = getState();
        const {am = {}, common = {}} = state;

        const response = await api.request('yasms.state', {track_id: getTrackId(state, TRACK_TYPES.COMMON)});

        if (response.status === 'ok') {
            const {account = {}} = response;
            const {phones} = account;

            if (phones) {
                const securePhonesIds = Object.keys(phones).filter((id) => phones[id].secured);
                const securePhoneId = securePhonesIds && securePhonesIds.length && securePhonesIds[0];
                const securePhoneNumber = (securePhoneId && phones[securePhoneId].number) || '';
                const {pathname} = new URL(common.currentUrl);

                dispatch(
                    setPreparedStore({
                        phoneNumber: (am.isAm && am.phone) || '',
                        securePhoneNumber,
                        securePhoneId,
                        errors: [],
                        confirmationCode: '',
                        isConfirmationCodeSent: false,
                        isCaptchaRequired: false,
                        isConfirmProcess: false,
                        isSecurePhoneMode: !am.isAm && am.mode === 'phoneconfirm' && !pathname.startsWith('/auth')
                    })
                );
            }
        }
    };
}

export const sendConfirmationCode = () => (dispatch, getState) => {
    const state = getState();
    const {phoneConfirm = {}, settings = {}, common = {}, am = {}} = state;
    const {phoneNumber, securePhoneNumber, isSecurePhoneMode, isConfirmProcess, securePhoneId} = phoneConfirm;
    const number = isConfirmProcess
        ? securePhoneNumber
        : phoneNumber || (am.isAm && am.mode === 'phoneconfirm' && am.phone);
    const {track_id, csrf} = common;
    const handler =
        isSecurePhoneMode && !isConfirmProcess
            ? 'phone/confirm_and_bind_secure/submit_v2'
            : 'phone/bind_simple_or_confirm_bound/submit';

    if (!number || !number.trim()) {
        dispatch(setErrors(['number.empty']));
        return;
    }

    const params = {
        number,
        track_id,
        csrf_token: csrf,
        display_language: settings.lang
    };

    const isKeyEnabled2FA = isKeyEnabled2FASelector(state);

    if (isKeyEnabled2FA && securePhoneId) {
        params.phone_id = securePhoneId;
    }

    if (!isAndroidAm(state)) {
        params.isCodeWithFormat = true;
    }

    dispatch(setErrors([]));
    dispatch(domikIsLoading(true));

    api.request(handler, params)
        .done((response = {}) => {
            dispatch(domikIsLoading(false));
            dispatch(changePagePopupVisibility(true));
            dispatch(changePagePopupType('phoneConfirmation'));
            dispatch({type: SET_PHONE_CONFIRM_TIMER});
            dispatch({
                type: SET_IS_PASSWORD_REQUIRED,
                payload: {
                    isPasswordRequired: response.isPasswordRequired
                }
            });
        })
        .fail((error = {}) => {
            const {errors} = error;

            if (errors.includes('phone.confirmed') || errors.includes('phone_secure.bound_and_confirmed')) {
                // sendMessage(SUCCESS_MESSAGE);
                dispatch(redirectToRetpath());
                return;
            }

            dispatch(setErrors(errors));
            dispatch(domikIsLoading(false));
        });
};

export const submitConfirmationCode = () => (dispatch, getState) => {
    const state = getState();
    const {phoneConfirm = {}, common = {}, am = {}, form = {}} = state;
    const {
        confirmationCode,
        isSecurePhoneMode,
        isConfirmProcess,
        isPasswordRequired,
        isCaptchaRequired,
        phoneNumber,
        securePhoneNumber
    } = phoneConfirm;
    const {values: {phoneCode} = {}} = form;
    const code = confirmationCode || phoneCode;
    const number = isConfirmProcess
        ? securePhoneNumber
        : phoneNumber || (am.isAm && am.mode === 'phoneconfirm' && am.phone);
    const {retpath, track_id, csrf} = common;
    const values = getFormValues(state);
    const handler =
        isSecurePhoneMode && !isConfirmProcess
            ? 'phone/confirm_and_bind_secure/commit_v2'
            : 'phone/bind_simple_or_confirm_bound/commit';

    let haveErrors = false;

    if (!code || !code.trim()) {
        dispatch(setErrors(['code.empty']));
        dispatch(
            updateErrors({
                field: FIELDS_NAMES.PHONE_CODE,
                error: getError(FIELDS_NAMES.PHONE_CODE, 'code.empty')
            })
        );
        haveErrors = true;
    }

    const params = {
        code,
        track_id,
        csrf_token: csrf
    };

    if (isPasswordRequired) {
        if (!values.password) {
            dispatch(
                updateErrors({
                    field: FIELDS_NAMES.PASSWORD,
                    error: getError(FIELDS_NAMES.PASSWORD, 'missingvalue_current')
                })
            );

            haveErrors = true;
        }

        params.password = values.password;
    }

    if (haveErrors) {
        dispatch(domikIsLoading(false));
        return;
    }

    dispatch(setErrors([]));
    dispatch(domikIsLoading(true));
    dispatch(updateErrors({field: FIELDS_NAMES.PHONE, error: {}}));
    dispatch(updateErrors({field: FIELDS_NAMES.PHONE_CODE, error: {}}));

    const isKeyEnabled2FA = isKeyEnabled2FASelector(state);

    const check = () =>
        api
            .request(handler, params)
            .done(async () => {
                if (am.isAm) {
                    dispatch(domikIsLoading(false));
                }

                if (isKeyEnabled2FA) {
                    await dispatch(authEnableKey2FA());
                }

                sendMessage(SUCCESS_MESSAGE, retpath);

                if (am.isAm && am.mode === 'phoneconfirm' && number) {
                    dispatch(amStorePhoneNumber(number));

                    return;
                }

                dispatch(redirectToRetpath());
            })
            .fail(async (error = {}) => {
                const {errors = []} = error;

                if (errors.includes('phone.confirmed') || errors.includes('phone_secure.bound_and_confirmed')) {
                    if (isKeyEnabled2FA) {
                        await dispatch(authEnableKey2FA());
                    }
                    dispatch(redirectToRetpath());
                    return;
                }

                const restErrors = [];

                errors.forEach((errorCode) => {
                    const isPasswordError = errorCode.startsWith('password');
                    const isCaptchaError = errorCode === 'captcha.required';

                    if (!isPasswordError && !isCaptchaError) {
                        restErrors.push(errorCode);
                    }

                    if (isPasswordError) {
                        dispatch(
                            updateErrors({
                                field: FIELDS_NAMES.PASSWORD,
                                error: getError(FIELDS_NAMES.PASSWORD, errorCode)
                            })
                        );
                    }

                    if (isCaptchaError) {
                        dispatch(setCaptchaRequired(true));
                    }
                });

                if (restErrors.length) {
                    dispatch(setErrors(restErrors));
                }

                dispatch(domikIsLoading(false));
                sendMessage(ERROR_MESSAGE, retpath);
            });

    if (isCaptchaRequired) {
        return dispatch(checkCaptcha({callback: check, action: setCaptchaRequired, actionArguments: [false]}));
    }

    check();
};

export const requestPhoneConfirmationCode = ({
    dispatch,
    getState,
    phoneValidateParams = {},
    trackId,
    phoneId,
    mode
}) => {
    const state = getState();
    const {am = {}} = state;
    const values = getFormValues(state);
    const language = getLanguage(state);
    const {phone} = values;
    const {isPhoneValidForCall, isPhoneValidForFlashCall} = phoneValidateParams;
    const phoneConfirmMethod = isPhoneValidForFlashCall || isPhoneValidForCall ? 'by_call' : 'by_sms';
    const track = trackId || getTrackId(state, 'register');

    dispatch(setPhoneConfirmationPhoneCheckState(false));
    dispatch(updateErrors({field: FIELDS_NAMES.PHONE, error: {}}));
    dispatch(updateErrors({field: FIELDS_NAMES.PHONE_CODE, error: {}}));
    dispatch(setPhoneConfirmMethod(phoneConfirmMethod));

    const requestParams = {trackId: track, language, phoneConfirmMethod};

    if (phoneId) {
        requestParams.phone_id = phoneId;
    } else {
        requestParams.phone = phone;
    }

    if (mode) {
        requestParams.mode = mode;
    }

    if (am.isAm && am.platform === Platforms.ANDROID) {
        requestParams.gpsPackageName = am.appId;
    }

    if (!isAndroidAm(state)) {
        requestParams.isCodeWithFormat = true;
    }

    return new Promise((resolve, reject) =>
        api
            .requestPhoneConfirmationCode(requestParams)
            .then((response = {}) => {
                const {
                    denyResendUntil = getUnixTimeStamp(30),
                    callingNumberTemplate,
                    internationalPhoneNumber,
                    phone,
                    isPasswordRequired
                } = response;

                dispatch(setPhoneConfirmationDenyResendUntilTime(denyResendUntil));
                dispatch(updatePhoneNumber(phone));
                dispatch(setPhoneConfirmationCallingNumberTemplate(callingNumberTemplate));
                dispatch(setPhoneConfirmationInternationalPhoneNumber(internationalPhoneNumber));
                dispatch(amRequestSmsCode());
                dispatch({
                    type: SET_IS_PASSWORD_REQUIRED,
                    payload: {
                        isPasswordRequired
                    }
                });

                api.checkPhoneConfirmationStatus({trackId: track, phoneConfirmMethod}).catch((error = {}) => {
                    const {code} = error;

                    if (code === 'check_phone') {
                        return dispatch(setPhoneConfirmationPhoneCheckState(true));
                    }

                    dispatch(
                        updateErrors({field: FIELDS_NAMES.PHONE_CODE, error: getError(FIELDS_NAMES.PHONE_CODE, code)})
                    );
                    metrika.send(`Показ ошибки поля phoneCode: ${code}`);
                });

                resolve(response);
            })
            .catch((error = {}) => {
                const code = error.errors && error.errors[0];

                if (code !== 'captcha.not_checked' && code !== 'captcha.required') {
                    dispatch(
                        updateErrors({
                            field: FIELDS_NAMES.PHONE,
                            error: getError(FIELDS_NAMES.PHONE, code)
                        })
                    );

                    dispatch(
                        updateErrors({
                            field: FIELDS_NAMES.PHONE_CODE,
                            error: getError(FIELDS_NAMES.PHONE, code)
                        })
                    );

                    metrika.send(`Показ ошибки поля phone: ${code}`);
                    metrika.send(`Показ ошибки поля phoneCode: ${code}`);
                }

                reject(error);
            })
    );
};

export const retryToRequestPhoneConfirmationCode = (trackId) => (dispatch, getState) => {
    requestPhoneConfirmationCode({dispatch, getState, trackId});
};
