import api from '@blocks/api';
import {getError, sendMessage} from '@blocks/utils';
import {getFormValues, getTrackId} from '@blocks/selectors';
import {TRACK_TYPES} from '@blocks/actions/tracks';
import {updateErrors, updateStates} from '@blocks/actions/form';
import {changeLoadingStatus} from '@blocks/actions/spin';
import metrics from '@blocks/metrics';

export const SET_PUBLIC_ID_UPDATED = 'SET_PUBLIC_ID_UPDATED';
export const TOGGLE_PUBLIC_ID = 'TOGGLE_PUBLIC_ID';
export const SET_PUBLIC_ID_VALUE = 'SET_PUBLIC_ID_VALUE';
export const UPDATE_NUMBER = 'UPDATE_NUMBER';
export const SET_PUBLIC_ID_ENABLED = 'SET_PUBLIC_ID_ENABLED';
export const SET_PUBLIC_ID_SUGGEST = 'SET_PUBLIC_ID_SUGGEST';
export const SET_PUBLIC_ID_SUGGEST_LOADING = 'SET_PUBLIC_ID_SUGGEST_LOADING';

export function setPublicIdStatus(payload) {
    return {
        type: SET_PUBLIC_ID_UPDATED,
        hasUpdated: payload
    };
}

export function setPublicIdVisible(payload = false) {
    return {
        type: TOGGLE_PUBLIC_ID,
        isOpened: payload
    };
}

export function setPublicIdValue(value = '') {
    return {
        type: SET_PUBLIC_ID_VALUE,
        value
    };
}

export function setPublicIdEnabled() {
    return {
        type: SET_PUBLIC_ID_ENABLED
    };
}

export function setPublicIdSuggest(publicIdSuggest = []) {
    return {
        type: SET_PUBLIC_ID_SUGGEST,
        publicIdSuggest
    };
}

export function setPublicIdSuggestLoading(value) {
    return {
        type: SET_PUBLIC_ID_SUGGEST_LOADING,
        value: Boolean(value)
    };
}

export function increaseNumberOfUpdates() {
    return {
        type: UPDATE_NUMBER
    };
}

export function validatePublicId() {
    return (dispatch, getState) => {
        const state = getState();
        const form = getFormValues(state);
        const data = {
            track_id: getTrackId(state, TRACK_TYPES.COMMON),
            id: form.publicId
        };

        dispatch(setPublicIdStatus(false));
        dispatch(updateStates({field: 'publicId', status: ''}));
        dispatch(changeLoadingStatus(true));

        api.request('validate-public-id', data)
            .done((response = {}) => {
                if (response.status === 'ok') {
                    dispatch(updateStates({field: 'publicId', status: 'valid'}));
                }
            })
            .fail((error = {}) => {
                const errorCode = Array.isArray(error.errors) && error.errors[0];

                metrics.send(['PublicId', 'Показ ошибки', errorCode]);

                if (errorCode === 'public_id.empty') {
                    return;
                }

                dispatch(updateErrors({field: 'publicId', error: getError('publicId', errorCode)}));
            })
            .always(() => {
                dispatch(changeLoadingStatus(false));
            });
    };
}

export function updatePublicId() {
    return (dispatch, getState) => {
        const state = getState();
        const {publicId} = state;
        const formValues = getFormValues(state);
        const data = {
            track_id: getTrackId(state, TRACK_TYPES.COMMON),
            public_id: formValues.publicId
        };

        dispatch(changeLoadingStatus(true));

        api.request('update.profile', data)
            .done((response = {}) => {
                if (response.status === 'ok') {
                    metrics.send(['PublicId', 'Успешное сохранение']);
                    sendMessage('success');
                    dispatch(setPublicIdStatus(true));
                    dispatch(increaseNumberOfUpdates());
                }
            })
            .fail((error = {}) => {
                const errorCode = (Array.isArray(error.errors) && error.errors[0]) || 'global';

                sendMessage('error');
                dispatch(updateErrors({field: 'publicId', error: getError('publicId', errorCode)}));
            })
            .always(() => {
                if (publicId.updatesNumber === 2) {
                    dispatch(
                        updateErrors({field: 'publicId', error: getError('publicId', 'public_id.update_not_allowed')})
                    );
                }

                dispatch(changeLoadingStatus(false));
            });
    };
}

export function savePublicId() {
    return (dispatch, getState) => {
        const formValues = getFormValues(getState());

        dispatch(setPublicIdEnabled());
        dispatch(setPublicIdValue(formValues.publicId));
    };
}

export function suggestPublicId() {
    return (dispatch, getState) => {
        const state = getState();
        const {header = {}, common = {}, publicId: publicIdState = {}, person = {}} = state;
        const {defaultAccount = {}} = header;
        const {login} = defaultAccount;
        const {firstname = '', lastname = ''} = person;
        const {track_id: trackId = ''} = common;
        const {showLoginInSuggest, id} = publicIdState;

        dispatch(setPublicIdSuggestLoading(true));

        api.request('suggestV2', {firstname, lastname, track_id: trackId})
            .done((response = {}) => {
                const {logins, status} = response;
                const suggest = logins && logins.length ? logins : [];
                const isLoginInSuggest = showLoginInSuggest && login !== id;

                if (status === 'ok') {
                    dispatch(setPublicIdSuggest(isLoginInSuggest ? [login, ...suggest] : suggest));
                }
            })
            .always(() => dispatch(setPublicIdSuggestLoading(false)));
    };
}
