import passport from '@plibs/pclientjs/js/passport';
import {showRequestPassword, showRequestCaptcha} from '@blocks/morda/components/domik/actions';
import {
    connectionError,
    GET_TRACK,
    GET_TRACK_FAIL,
    GET_TRACK_SUCCESS,
    goToAuth,
    updateCSRF
} from '@blocks/common/actions';
import {TRACK_TYPES, updateTrackId} from '@blocks/actions/tracks';
import {isMobile, getProcessUUid, getUid} from '@blocks/selectors';
import {getQueryParam} from '@blocks/utils';

let store;

export default {
    init(s) {
        store = s;
    },

    request(method, params = {}, options) {
        const deferred = new $.Deferred();

        passport.api
            .request(method, params, options)
            .done((res) => {
                if (res.status !== 'error') {
                    return deferred.resolve(res);
                }

                if (Array.isArray(res.errors)) {
                    if (
                        res.errors.find((err) => err === 'password.required') &&
                        !['devices.revoke-tokens'].includes(method)
                    ) {
                        store.dispatch(showRequestPassword(true));
                    }
                    if (res.errors.find((err) => err === 'captcha.required') && method !== '/profile/password/submit') {
                        store.dispatch(showRequestCaptcha(true));
                    }
                    if (res.errors.find((err) => err === 'sessionid.invalid')) {
                        store.dispatch(goToAuth());
                    }
                    if (res.errors.find((err) => err === 'track.not_found' || err === 'track_id.invalid')) {
                        this.getTrackWithUid();
                    }
                }

                return deferred.reject(res);
            })
            .fail((jqXHR) => {
                const status = jqXHR.status;
                const result = {
                    status: 'error',
                    errors: []
                };

                if (status === 403) {
                    result.errors.push('csrf_token.invalid');
                } else {
                    result.errors.push('internal');
                }

                store.dispatch(connectionError(result));
                return deferred.reject(result);
            });

        return deferred.promise();
    },

    log(message, options) {
        passport.api.log(message, options);
    },

    async doCompleteSubmit(data) {
        return await this.request('do-complete-submit', data);
    },

    getTrackWithUid() {
        const state = store.getState();
        const common = state.common;
        const csrf = common.csrf;
        const actionForRepeat = common.actionForRepeat || {};
        const {action, payload} = actionForRepeat;

        store.dispatch({type: GET_TRACK});

        this.request('getTrackWithUid', {csrf_token: csrf})
            .done((result) => {
                store.dispatch({
                    type: GET_TRACK_SUCCESS,
                    track_id: result.track_id
                });

                if (action) {
                    store.dispatch(action(...payload));
                }
            })
            .fail((res) => {
                store.dispatch({
                    type: GET_TRACK_FAIL,
                    res
                });
            });
    },

    getTrack({trackId, type}) {
        return new Promise((resolve) => {
            this.request('track', {
                type,
                // потому что ручка трека хочет трек ᕕ( ᐛ )ᕗ
                track_id: trackId
            })
                .done((response = {}) => {
                    const {id} = response;

                    store.dispatch(updateTrackId({id, type}));
                    resolve(response);
                })
                .fail(resolve);
        });
    },

    setCsrfToken(csrfToken) {
        passport.api.setCsrfToken(csrfToken);
    },

    validatePhoneById({phoneId}) {
        const requestParams = {
            validate_for_call: true,
            phoneId
        };

        return new Promise((resolve) => {
            this.request('auth/validate_phone_by_id', requestParams)
                .done((response = {}) => {
                    const {
                        valid_for_call: isPhoneValidForCall,
                        valid_for_flash_call: isPhoneValidForFlashCall
                    } = response;

                    resolve({isPhoneValidForCall, isPhoneValidForFlashCall});
                })
                .fail(resolve);
        });
    },

    validatePhone({trackId, phone}) {
        const requestParams = {
            track_id: trackId,
            validate_for_call: true,
            phone_number: phone
        };

        return new Promise((resolve) => {
            this.request('validate-phone', requestParams)
                .done((response = {}) => {
                    const {
                        valid_for_call: isPhoneValidForCall,
                        valid_for_flash_call: isPhoneValidForFlashCall
                    } = response;

                    resolve({isPhoneValidForCall, isPhoneValidForFlashCall});
                })
                .fail(resolve);
        });
    },

    requestPhoneConfirmationCode({
        trackId,
        language,
        phone,
        phone_id,
        phoneConfirmMethod,
        mode,
        gpsPackageName,
        isCodeWithFormat
    }) {
        const requestParams = {
            track_id: trackId,
            display_language: language,
            number: phone,
            phone_id,
            confirm_method: phoneConfirmMethod,
            mode,
            gps_package_name: gpsPackageName,
            isCodeWithFormat
        };

        return new Promise((resolve, reject) => {
            this.request('phone-confirm-code-submit', requestParams)
                .done((response = {}) => {
                    const {
                        deny_resend_until: denyResendUntil,
                        calling_number_template: callingNumberTemplate,
                        number: {
                            international: internationalPhoneNumber,
                            masked_international: maskedInternational
                        } = {},
                        is_password_required: isPasswordRequired = false
                    } = response;

                    resolve({
                        denyResendUntil,
                        callingNumberTemplate,
                        phone: maskedInternational,
                        internationalPhoneNumber,
                        isPasswordRequired
                    });
                })
                .fail(reject);
        });
    },

    checkPhoneConfirmationStatus({trackId, phoneConfirmMethod}) {
        return new Promise((resolve, reject) => {
            const _checkPhoneConfirmationStatus = () =>
                setTimeout(() => {
                    this.request('confirm-phone-check-status', {track_id: trackId})
                        .then() // do nothing
                        .fail((error = {}) => {
                            const {errors = []} = error;
                            const errorCode = errors[0] || 'call_confirm.failed';
                            const shouldSkipError = ['call_confirm.not_ready'].includes(errorCode);
                            const isCallConfirmInProgressOrFinished = [
                                'call_confirm.in_progress',
                                'call_confirm.finished',
                                'call_confirm.started'
                            ].includes(errorCode);
                            const shouldCheckPhone = ['call_confirm.failed', 'call_confirm.user_unavailable'].includes(
                                errorCode
                            );

                            if (!['by_flash_call', 'by_call'].includes(phoneConfirmMethod)) {
                                return;
                            }

                            if (errorCode === 'track.invalid_state') {
                                // Невалидным трек может стать если попросили отправить СМС во время звонка
                                // и в этом кейсе будет нормально не показать ошибку.
                                // В противном случае у пользователя есть возможно попробовать отправить СМС
                                // и увидеть акутальную ошибку.
                                return;
                            }

                            if (phoneConfirmMethod === 'by_flash_call' && errorCode === 'call_confirm.user_hangup') {
                                // нормальное поведение для этого кейса
                                return;
                            }

                            if (shouldSkipError) {
                                return _checkPhoneConfirmationStatus();
                            }

                            if (isCallConfirmInProgressOrFinished) {
                                return resolve();
                            }

                            if (shouldCheckPhone) {
                                return reject({code: 'check_phone'});
                            }

                            return reject({code: errorCode});
                        });
                }, 500);

            if (['by_flash_call', 'by_call'].includes(phoneConfirmMethod)) {
                _checkPhoneConfirmationStatus();
            }
        });
    },

    checkPhoneConfirmationCode({trackId, code, mode, password}) {
        return new Promise((resolve, reject) => {
            this.request('phone-confirm-code', {track_id: trackId, code, password, mode})
                .done(resolve)
                .fail(reject);
        });
    },

    checkPhoneBySquatter({csrf, trackId, phone, scenario}) {
        return new Promise((resolve, reject) => {
            this.request('phone-validate-by_squatter', {
                phone,
                scenario,
                track_id: trackId,
                csrf_token: csrf
            })
                .done(resolve)
                .fail(reject);
        });
    },

    findAccountsByNameAndPhone({trackId, firstname, lastname, allowNeophonish}) {
        return new Promise((resolve, reject) => {
            this.request('find-accounts-by-name-and-phone', {
                allow_neophonish: allowNeophonish,
                track_id: trackId,
                firstname,
                lastname
            })
                .done(resolve)
                .fail(reject);
        });
    },
    async findAccountsByPhone({trackId}) {
        return await this.request('find-accounts-by-phone', {
            track_id: trackId
        });
    },

    getSuggestedLogins({trackId, firstname, lastname, login}) {
        const requestParams = {
            track_id: trackId,
            firstname,
            lastname,
            login,
            timeout: 2000
        };

        return new Promise((resolve, reject) => {
            if (!requestParams.firstname.trim() || !requestParams.lastname.trim()) {
                resolve([]);
            }

            this.request('suggestV2', requestParams)
                .done(resolve)
                .fail(reject);
        });
    },

    registrationSubmit() {
        return new Promise((resolve) => {
            this.request('registration-submit')
                .done((response = {}) => {
                    const {id} = response;

                    store.dispatch(updateTrackId({id, type: TRACK_TYPES.REGISTER}));
                    resolve(response);
                })
                .fail(resolve);
        });
    },

    multiStepCommitSMSCode({trackId, login, retpath}) {
        return new Promise((resolve, reject) => {
            this.request('multi-step-commit-sms-code', {track_id: trackId, login, retpath})
                .done(resolve)
                .fail(reject);
        });
    },

    checkJSLoad({trackId, language}) {
        this.request('checkjsload', {
            track_id: trackId,
            language
        });
    },

    writeStatbox(stat) {
        const state = store.getState();
        const {common = {}} = state;
        const {experiments = {}, retpath, origin, from} = common;
        const {flagsString, boxes} = experiments;
        const {pathname} = typeof window !== undefined ? location : {};

        const message = Object.assign(
            {
                experiment_flags: flagsString,
                experiment_boxes: boxes && boxes.join(';'),
                is_mobile: `${isMobile(state)}`,
                pathname,
                url: pathname,
                retpath: retpath && encodeURI(retpath),
                uid: getUid(state),
                process_uuid: getProcessUUid(state),
                origin: origin && encodeURIComponent(origin),
                from: from && encodeURIComponent(from),
                login: getQueryParam('login')
            },
            stat
        );

        passport.api.log(message, {
            encrypt: true
        });
    },

    getAccounts({trackId}) {
        return new Promise((resolve, reject) => {
            const {common: {origin} = {}} = store.getState();

            this.request('auth/accounts', {track_id: trackId, origin})
                .done(resolve)
                .fail(reject);
        });
    },

    askAdditionalData({uid}) {
        return new Promise((resolve, reject) => {
            this.request('auth/additional_data/ask_v2', {uid})
                .done(resolve)
                .fail(reject);
        });
    },

    updateCsrfToken() {
        return new Promise((resolve, reject) => {
            this.request('update-csrf')
                .done((response = {}) => {
                    const {updatedCSRF} = response;

                    if (updatedCSRF) {
                        this.setCsrfToken(updatedCSRF);
                        store.dispatch(updateCSRF(updatedCSRF));
                    }

                    resolve(response);
                })
                .fail(reject);
        });
    },

    userEntryFlowSubmit({process} = {}) {
        return new Promise((resolve, reject) => {
            const state = store.getState();
            const {common = {}, am = {}} = state;
            const {track_id} = common;
            const {isAm, appId} = am;

            this.request('user-entry-flow-submit', {track_id, isAm, appId, process})
                .done((response = {}) => {
                    const {id} = response;

                    store.dispatch(updateTrackId({id, type: TRACK_TYPES.REGISTER}));
                    resolve(response);
                })
                .fail(reject);
        });
    },

    neoPhonishAuth({trackId, firstname, lastname, uid, retpath}) {
        return new Promise((resolve) => {
            this.request('neo-phonish-auth', {track_id: trackId, firstname, lastname, uid, retpath})
                .done(resolve)
                .fail(resolve);
        });
    },

    neoPhonishRestore({trackId, uid}) {
        return new Promise((resolve) => {
            this.request('neo-phonish-auth', {track_id: trackId, uid, useNewSuggestByPhone: true})
                .done(resolve)
                .fail(resolve);
        });
    },

    getRegisterLiteState({trackId, language}) {
        store.dispatch(updateTrackId({id: trackId, type: TRACK_TYPES.REGISTER}));

        return new Promise((resolve, reject) => {
            this.request('registration-lite-experiment.submit', {track_id: trackId, language})
                .done(resolve)
                .fail(reject);
        });
    },

    registerLite(params) {
        return new Promise((resolve, reject) => {
            this.request('registration-lite-experiment.commit', params)
                .done(resolve)
                .fail(reject);
        });
    },

    registerSocial(params) {
        return new Promise((resolve, reject) => {
            this.request('registration-social', params)
                .done(resolve)
                .fail(reject);
        });
    },

    socialAuthCallback() {
        return new Promise((resolve, reject) => {
            this.request('callback-social')
                .done(resolve)
                .fail(reject);
        });
    }
};
