import PropTypes from 'prop-types';
import React from 'react';
import classnames from 'classnames';

import {Button} from '@components/Button';
import {Input} from '@components/Input';

import CaptchaField from '../../components/captcha_field.jsx';
import Timer from './timer.jsx';
import {additionalDataRequestErrors as errorMessages} from '../../errors';

export default class EnterConfirmationCode extends React.Component {
    constructor(props) {
        super(props);

        this.startTimer = this.startTimer.bind(this);
        this.stopTimer = this.stopTimer.bind(this);

        this.timerInterval = null;
        this.state = {currentTimestamp: Number(new Date())};
    }

    startTimer() {
        this.timerInterval = setInterval(() => {
            const currentTimestamp = Number(new Date());

            this.setState(() => ({currentTimestamp}));

            if (currentTimestamp >= Number(this.props.denyResendUntil)) {
                this.stopTimer();
            }
        }, 1000);
    }

    stopTimer() {
        if (this.timerInterval) {
            clearInterval(this.timerInterval);
            this.timerInterval = null;
        }
    }

    componentWillUpdate(nextProps) {
        if (this.props.denyResendUntil !== nextProps.denyResendUntil) {
            this.setState(() => ({currentTimestamp: Number(new Date())}));
            this.startTimer();
        }
    }

    componentDidMount() {
        this.stopTimer();
        this.startTimer();
    }

    componentWillUnmount() {
        this.stopTimer();
    }

    render() {
        const {
            handleConfirmationCodeChange,
            approvePhone,
            sentConfirmationCode,
            goBack,
            goEnterPhone,
            confirmationCode,
            phoneNumber,
            isCaptchaRequired,
            denyResendUntil,
            action,
            errors
        } = this.props;
        const {currentTimestamp} = this.state;
        const fieldName = 'confirmation_code';
        const showTimer = currentTimestamp + 1000 < Number(denyResendUntil); // не показывать нулевой таймер

        return (
            <form className='enter-confirmation-code_form'>
                <h1 className='request-phone_title'>
                    {i18n('AdditionalDataRequest.phone.enter-confirmation-code.title')}
                </h1>
                <p
                    className='request-phone_description'
                    dangerouslySetInnerHTML={{
                        __html: i18n(
                            'AdditionalDataRequest.phone.enter-confirmation-code.description',
                            '<b>',
                            phoneNumber,
                            '</b>'
                        )
                    }}
                />

                {action !== 'secure' && (
                    <p className='change-email-link'>
                        <span onClick={goEnterPhone} role='button'>
                            {i18n('AdditionalDataRequest.phone.change-link')}
                        </span>
                    </p>
                )}

                <div
                    className={classnames('request-phone_form-field form__field', {
                        form__field_filled: confirmationCode
                    })}
                >
                    <Input
                        size='l'
                        id={fieldName}
                        name={fieldName}
                        autoFocus={true}
                        value={confirmationCode}
                        onChange={handleConfirmationCodeChange}
                    />
                    <label htmlFor={fieldName} className='form__field-label'>
                        {i18n('_AUTH_.phone.enter-confirmation-code.label')}
                    </label>
                </div>

                {isCaptchaRequired && <CaptchaField />}

                {errors.length !== 0 && (
                    <p className='request-phone_error'>
                        {errorMessages[errors[0]] ? errorMessages[errors[0]] : errorMessages.internal}
                    </p>
                )}

                <div className='request-phone_approve-button'>
                    <Button view='action' size='l' width='max' type='submit' onClick={approvePhone}>
                        {i18n('_AUTH_.phone.enter-confirmation-code.submit')}
                    </Button>
                </div>

                <div className='request-phone_send-again-button'>
                    <Button
                        view='pseudo'
                        size='l'
                        width='max'
                        type='button'
                        disabled={showTimer}
                        onClick={sentConfirmationCode}
                    >
                        {i18n('AdditionalDataRequest.phone.enter-confirmation-code.send-again')}
                        {showTimer && <Timer startTime={currentTimestamp} endTime={Number(denyResendUntil)} />}
                    </Button>
                </div>

                <div className='request-phone_skip-button'>
                    <Button view='pseudo' size='l' width='max' type='button' onClick={goBack}>
                        {i18n('_AUTH_.phone.enter-confirmation-code.skip')}
                    </Button>
                </div>
            </form>
        );
    }
}

EnterConfirmationCode.propTypes = {
    handleConfirmationCodeChange: PropTypes.func.isRequired,
    approvePhone: PropTypes.func.isRequired,
    sentConfirmationCode: PropTypes.func.isRequired,
    goBack: PropTypes.func.isRequired,
    goEnterPhone: PropTypes.func.isRequired,
    confirmationCode: PropTypes.string.isRequired,
    action: PropTypes.string.isRequired,
    phoneNumber: PropTypes.string.isRequired,
    isCaptchaRequired: PropTypes.bool.isRequired,
    denyResendUntil: PropTypes.number.isRequired,
    errors: PropTypes.array.isRequired
};
