import {push} from 'connected-react-router';
import api from '@blocks/api';
import {domikIsLoading, setPasswordError, changeCaptchaState, updatePasswordValue} from './';
import {updateCSRF} from '@blocks/common/actions';
import {preparePhoneConfirmStore} from '@blocks/actions/phoneConfirm';
import {getRedirectUrl, isKeyEnabled2FASelector} from '@blocks/selectors';
import reloadCaptcha from '@components/Captcha/actions/reloadCaptcha';
import additionalDataAsk from './additionalDataAsk';
import changeNativeInputValue from './changeNativeInputValue';
import sendToChallenge from './challenge/sendToChallenge';
import {ACCOUNT_TYPES, getAccountTypeByAlias} from '@blocks/authv2/utils/accountTypes';
import {
    SHOW_PASSWORD_FORM,
    SHOW_SOCIAL_WITH_PASSWORD_FORM,
    SHOW_OTP_FORM,
    SEND_AUTH_FORM,
    SEND_AUTH_FORM_SUCCESS,
    SEND_AUTH_FORM_ERROR,
    SUCCESS_AUTH_GOAL
} from '@blocks/authv2/metrics_constants';
import {submitAuthSucceed, amSetAnalyticsToRetpath} from '@blocks/authv2/actions/nativeMobileApi';
import metrics from '@blocks/metrics';
import sendToForcePasswordChange from './forcePasswordChange/sendToForcePasswordChange';

const SOCIAL_AUTH_METHODS = [
    'social_gg',
    'social_fb',
    'social_tw',
    'social_mr',
    'social_vk',
    'social_ok',
    'social_esia'
];

export default function multiStepAuthCommitPassword(password) {
    return (dispatch, getState) => {
        const state = getState();
        const {
            common: {track_id: trackId, csrf, origin, phoneConfirmUrl, preRegCompleteUrl},
            auth: {
                form: {isForceOTP, password: passwordFromState},
                processedAccount: account
            },
            am = {}
        } = state;

        let redirectUrl = getRedirectUrl(state);

        if (am.isAm) {
            if (typeof redirectUrl === 'string' && redirectUrl.indexOf('from=captcha') === -1) {
                dispatch(amSetAnalyticsToRetpath(am.isAuthBySWC ? 'Smartlock' : 'Login'));
                redirectUrl = getRedirectUrl(getState());
            }
        }

        const allowedAuthMethods = (account && account.allowed_auth_methods) || [];

        let authMethod = (account && account.preferred_auth_method) || 'password';

        if (allowedAuthMethods.indexOf('otp') > -1 && isForceOTP) {
            authMethod = 'otp';
        }

        const params = {
            track_id: trackId,
            csrf_token: csrf,
            password: password || passwordFromState,
            retpath: redirectUrl
        };

        dispatch(domikIsLoading(true));

        api.request('auth/multi_step/commit_password', params)
            .done((response) => {
                if (!response) {
                    dispatch(domikIsLoading(false));
                    return;
                }

                const {
                    retpath,
                    status,
                    state: responseState,
                    redirect_url: responseRedirectUrl,
                    avatarId = null
                } = response;
                const isAM = retpath && retpath.indexOf('am_social=true') !== -1;

                const finish = () => {
                    metrics.goal(SUCCESS_AUTH_GOAL);
                    metrics.send([getMetricsHeader(state), SEND_AUTH_FORM, SEND_AUTH_FORM_SUCCESS]);

                    if (am.isAm) {
                        dispatch(domikIsLoading(false));
                    }

                    if (status === 'ok' && !responseState) {
                        if (isAM) {
                            return (window.location = retpath || redirectUrl);
                        }

                        dispatch(additionalDataAsk());
                    }

                    if (responseState === 'auth_challenge') {
                        dispatch(sendToChallenge());
                        return;
                    }

                    if (responseState === 'change_password') {
                        dispatch(sendToForcePasswordChange());
                        return;
                    }

                    if (responseState && responseRedirectUrl) {
                        return (window.location = responseRedirectUrl);
                    }
                };

                if (
                    am.isAm &&
                    status === 'ok' &&
                    !responseState &&
                    (account.preferred_auth_method || 'password') === 'password' &&
                    account.login &&
                    account.login !== account.phone
                ) {
                    if (!avatarId) {
                        dispatch(domikIsLoading(true));

                        const isKeyEnabled2FA = isKeyEnabled2FASelector(state);

                        api.request('auth/accounts', {origin})
                            .done(async ({accounts = {}, csrf}) => {
                                api.setCsrfToken(csrf);
                                dispatch(updateCSRF(csrf));

                                dispatch(domikIsLoading(false));

                                const processedAccount = accounts.processedAccount || {};
                                const {avatarId = '0/0-0'} = processedAccount;

                                account.avatarId = avatarId;

                                if (isKeyEnabled2FA) {
                                    if (
                                        getAccountTypeByAlias(
                                            processedAccount.primaryAliasType || account.primaryAliasType
                                        ) !== ACCOUNT_TYPES.PORTAL
                                    ) {
                                        return dispatch(push(preRegCompleteUrl));
                                    }
                                    await dispatch(preparePhoneConfirmStore());

                                    return dispatch(push(phoneConfirmUrl));
                                } else {
                                    dispatch(submitAuthSucceed(account, password, finish));
                                }
                            })
                            .fail(async () => {
                                dispatch(domikIsLoading(false));
                                if (isKeyEnabled2FA) {
                                    if (getAccountTypeByAlias(account.primaryAliasType) !== ACCOUNT_TYPES.PORTAL) {
                                        return dispatch(push(preRegCompleteUrl));
                                    }
                                    await dispatch(preparePhoneConfirmStore());

                                    return dispatch(push(phoneConfirmUrl));
                                } else {
                                    dispatch(submitAuthSucceed(account, password, finish));
                                }
                            });

                        return;
                    }
                    account.avatarId = avatarId;

                    return dispatch(submitAuthSucceed(account, password, finish));
                }

                finish();
            })
            .fail((result = {}) => {
                const error = result.errors[0];
                const {
                    auth: {isCaptchaRequired}
                } = state;

                metrics.send([getMetricsHeader(state), SEND_AUTH_FORM, SEND_AUTH_FORM_ERROR, error]);

                if (['captcha.required', 'captcha.invalid'].indexOf(error) > -1) {
                    dispatch(changeCaptchaState(true));

                    if (isCaptchaRequired) {
                        dispatch(reloadCaptcha());
                    }
                }

                if (
                    ['password.not_matched', 'password.empty', 'rfc_otp.invalid'].includes(error) ||
                    (authMethod === 'otp' && ['captcha.required', 'captcha.invalid'].includes(error))
                ) {
                    dispatch(updatePasswordValue(''));
                    dispatch(changeNativeInputValue('passwd', ''));
                }

                dispatch(setPasswordError(error));
                dispatch(domikIsLoading(false));
            });
    };
}

export function getMetricsHeader(state) {
    const {
        auth: {
            form: {isForceOTP},
            processedAccount: account
        },
        social: {providers}
    } = state;
    const allowedAuthMethods = (account || {}).allowed_auth_methods || [];

    let authMethod = (account || {}).preferred_auth_method || 'password';

    if (allowedAuthMethods.indexOf('otp') > -1 && isForceOTP) {
        authMethod = 'otp';
    }

    const provider = providers.filter(({data}) => `social_${data.code}` === authMethod)[0] || null;
    const isShowPasswordField = Boolean(authMethod === 'password' || allowedAuthMethods.indexOf('password') > -1);
    const hasSocialButton = Boolean(SOCIAL_AUTH_METHODS.indexOf(authMethod) > -1 && provider);

    if (isShowPasswordField && hasSocialButton) {
        return SHOW_SOCIAL_WITH_PASSWORD_FORM;
    }

    if (isShowPasswordField) {
        return SHOW_PASSWORD_FORM;
    }

    return SHOW_OTP_FORM;
}
