import filter from 'lodash/filter';

import {hasExp} from '@blocks/selectors';
import {setTypeImage2FA} from '@blocks/authv2/actions';
import {TYPE_2FA_AUTH} from '@blocks/authv2/constants';
import api from '../../api';
import {
    changeProcessedAccount,
    update2faPictures,
    updateTokensSuccess,
    domikIsLoading,
    setLoginError,
    updateLoginValue
} from './';

export default function multiStepAuthStart(login) {
    return (dispatch, getState) => {
        const state = getState();
        const {
            common: {csrf, retpath, fretpath, origin, from: service},
            auth: {
                process_uuid: processUUID,
                form: {pdd_domain: pddDomain},
                processedAccount,
                typeImage2FA
            }
        } = getState();
        const params = {
            csrf_token: csrf,
            login
        };
        const authMethods = (processedAccount && processedAccount.allowed_auth_methods) || [];
        const isAccountWith2FA = authMethods.includes('magic') && authMethods.includes('otp');

        if (isAccountWith2FA && hasExp(state, 'is-2fa-with-image') && typeImage2FA === TYPE_2FA_AUTH.IMAGE) {
            params.with_2fa_pictures = 1;
        }

        if (processUUID) {
            params.process_uuid = processUUID;
        }

        if (retpath) {
            params.retpath = retpath;
        }

        if (fretpath) {
            params.fretpath = fretpath;
        }

        if (origin) {
            params.origin = origin;
        }

        if (service) {
            params.service = service;
        }

        if (pddDomain && login && !/@/.test(login)) {
            params.login = `${params.login}@${pddDomain}`;
        }

        if (pddDomain) {
            params.is_pdd = '1';
        }

        dispatch(domikIsLoading(true));
        dispatch(setLoginError(null));
        dispatch(updateLoginValue(login));

        api.request('auth/multi_step/start', params)
            .done((response) => {
                const {
                    auth_methods: authMethods,
                    preferred_auth_method: preferredAuthMethod,
                    track_id: trackId,
                    csrf_token: magicCSRF,
                    magic_link_email: email,
                    '2fa_pictures': auth2faPictures
                } = response;

                if (auth2faPictures) {
                    dispatch(setTypeImage2FA(TYPE_2FA_AUTH.IMAGE));
                    dispatch(update2faPictures(auth2faPictures));
                }

                const {
                    auth: {unitedAccounts, isQRAuthEnabled}
                } = getState();

                let processedAccount = filter(unitedAccounts, (account) => account.login === login)[0];

                const firstAuthMethod = (authMethods && authMethods[0]) || 'password';

                if (!processedAccount) {
                    processedAccount = {
                        isAddedAccount: true,
                        allowed_auth_methods: authMethods,
                        preferred_auth_method: firstAuthMethod,
                        avatarId: '0/0-0',
                        email,
                        login
                    };
                } else {
                    processedAccount = Object.assign({}, processedAccount, {
                        allowed_auth_methods: processedAccount.allowed_auth_methods || authMethods || [],
                        preferred_auth_method:
                            processedAccount.preferred_auth_method || preferredAuthMethod || firstAuthMethod,
                        email
                    });
                }

                const authMethod = processedAccount.preferred_auth_method;

                // Если выключен эксперимент на авторизацию по QR-коду, но пришел метод авторизации magic_x_token, то
                // необходимо выбрать следующий достуный способ авторизации.
                // И на всякий случай сделать fallback на авторизацию по паролю
                // TODO: убрать, когда эксперимент выкатим на 100%
                if (authMethod === 'magic_x_token' && !isQRAuthEnabled) {
                    const allowedAuthMethods = processedAccount.allowed_auth_methods;

                    processedAccount.preferred_auth_method =
                        allowedAuthMethods.filter((method) => method !== 'magic_x_token')[0] || 'password';
                }

                dispatch(changeProcessedAccount(processedAccount));
                dispatch(updateTokensSuccess(trackId, magicCSRF));
            })
            .fail(() => {
                // Do nothing, auth magic will be OK even if multi_step/start fails
            })
            .always(() => {
                dispatch(domikIsLoading(false));
            });
    };
}
